# -*- coding: iso-8859-1 -*-
# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
#              C E D A R
#          S O L U T I O N S       "Software done right."
#           S O F T W A R E
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Copyright (c) 2002-2003 Kenneth J. Pronovici.
# All rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# Version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Copies of the GNU General Public License are available from
# the Free Software Foundation website, http://www.gnu.org/.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.2)
# Project  : Cedar Backup
# Revision : $Id: cdr.py,v 1.14 2002/09/20 01:41:38 pronovic Exp $
# Purpose  : Provides CD-R functionality for the project
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

# This file was created with a width of 132 characters, and NO tabs.

########
# Notes
########

# This file is not intended to be an executable script.  Instead, it
# is a Python module file that provides common functionality for Cedar
# Backup scripts.  Import this file to use the functionality.


######################
# Pydoc documentation
######################

"""
Provides CD-R and CD-RW (writable compact disc) functionality.

Functions that start with _ should be considered private to this
module, and should not be used by code outside this module.
"""

__author__  = "Kenneth J. Pronovici"


########################################################################
# Imported modules
########################################################################

# System modules
import os
import re
import exceptions

# Cedar Backup modules
from CedarBackup.exceptions import CedarBackupError
from CedarBackup.filesystem import execute_command


#######################################################################
# Module-wide configuration and constants
#######################################################################

# External programs
CDRECORD_PROGRAM = 'cdrecord'
EJECT_PROGRAM    = 'eject'
MKISOFS_PROGRAM  = 'mkisofs'
MOUNT_PROGRAM    = 'mount'


#######################################################################
# Media type definitions
#######################################################################

# Sector conversion constants
BYTES_PER_KBYTE   = 1024.0
KBYTES_PER_MBYTE  = 1024.0
BYTES_PER_SECTOR  = 2048.0
BYTES_PER_MBYTE   = BYTES_PER_KBYTE * KBYTES_PER_MBYTE
KBYTES_PER_SECTOR = BYTES_PER_SECTOR / BYTES_PER_KBYTE
MBYTES_PER_SECTOR = BYTES_PER_SECTOR / BYTES_PER_MBYTE

# These are the media types we support
MEDIA_TYPES = [ 'cdrw-74', 'cdr-74' ]

# These are the attributes for the media types we support
MEDIA_ATTRIBUTES = { 'cdrw-74' : { 'writable' : True,  
                                   'initial_lead_in': 11400,      # in 2k-blocks
                                   'lead_in': 6900,               # in 2k-blocks
                                   'capacity' : { 'bytes'  : 681574400, 
                                                  'kbytes' : 665600, 
                                                  'blocks' : 332800, 
                                                  'mbytes' : 650 } },
                     'cdr-74' : { 'writable' : False,
                                  'initial_lead_in': 11400,       # in 2k-blocks
                                  'lead_in': 6900,                # in 2k-blocks
                                  'capacity' : { 'bytes'  : 681574400, 
                                                 'kbytes' : 665600, 
                                                 'blocks' : 332800, 
                                                 'mbytes' : 650 } }       }
                                   

#######################################################################
# Public functions
#######################################################################

########################
# initialize() function
########################

def initialize(device, scsi_id, drive_speed, media_type):

   ######################
   # Pydoc documentation
   ######################

   """
   Initialize a CD-R or CD-RW device.

   Arguments:
      
      - **device** : Filesystem device name (i.e. '/dev/cdrw')

      - **scsi_id** : SCSI id for the device (i.e. '[devicename, scsibus, lun]')

      - **drive_speed** : Speed at which drive writes (i.e. '2' for 2x drive, etc.)

      - **media_type** : One of 'cdr-74' or 'cdrw-74'

   Returns a dictionary of results:

      - 'device' : Dictionary describing the results or None

      - 'warnings' : List of warnings about the device or None

      - 'errors' : List of errors encountered or None

   If there are any errors at all, the 'device' entry will be None.  

   The contents of the returned 'device' entry should be considered
   private to this package.
   """

   #######################
   # Initialize variables
   #######################

   results             = { }
   results['device']   = None
   results['errors']   = None
   results['warnings'] = None

   device_dict         = { }
   errors              = []
   warnings            = []

 
   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ##########################
      # Validate the media type
      ##########################

      if not media_type in MEDIA_TYPES:
         errors.append("Error: unknown media type '%s'.\n" % media_type)
         raise CedarBackupError()

      device_dict['media_type']   = media_type


      ###########################
      # Validate the drive speed
      ###########################

      if drive_speed < 1:
         errors.append("Error: drive speed '%d' invalid; must be an integer >= 1.\n" % drive_speed)
         raise CedarBackupError()

      device_dict['drive_speed']  = drive_speed


      ################################
      # Validate the scsi_id argument
      ################################

      if len(scsi_id) < 3:
         errors.append("Error: SCSI id '%s' invalid; must be in the form [scsibus,target,lun].\n" % scsi_id)
         raise CedarBackupError()

      try:
         device_dict['devicename']  = int(scsi_id[0])
         device_dict['scsibus']     = int(scsi_id[1])
         device_dict['lun']         = int(scsi_id[2])
      except:
         errors.append("Error: SCSI devicename,scsibus,lun (%s) must be integers." % scsi_id);
         raise CedarBackupError()

      if device_dict['devicename'] < 0:
         errors.append("Error: SCSI devicename '%d' invalid; value must be >= 0.\n" % device_dict['devicename'])
         raise CedarBackupError()

      if device_dict['scsibus'] < 0:
         errors.append("Error: SCSI scsibus value '%d' invalid; must be >= 0.\n" % device_dict['scsibus'])
         raise CedarBackupError()

      if device_dict['lun'] < 0:
         errors.append("Error: lun value '%s' invalid; must be >= 0.\n" % device_dict['lun'])
         raise CedarBackupError()


      ######################
      # Validate the device
      ######################

      # This method of checking is UNIX-specific.  I guess I don't care
      # for now.
      #
      # The first thing we do is make sure the device appears to be
      # writable.  Then, we attempt to check that the device is not
      # currently mounted.  If it is mounted, that will cause us
      # problems.
      #
      # The mount check is a little bit complicated - we can easily
      # check whether (for instance) /dev/cdrw is mounted by using 'df
      # -k' or 'mount -l'.  However, what if /dev/cdrw is a soft-link
      # and the real device is the one that's mounted?  We're going to
      # ignore that difficulty for the time being, because the check
      # could easily get recursive.  In any case, these are just checks
      # for obvious problems - just because these checks pass doesn't
      # necessarily mean that the write will succeed.

      if not os.access(device, os.W_OK):
         errors.append("Error: device does not appear to be writable by effective user.\n")
         raise CedarBackupError()

      mount_pattern = re.compile(r"(^)(%s)(.*)($)" % device)
      mount_cmd = "%s -l" % MOUNT_PROGRAM
      (result, lines) = execute_command(mount_cmd)
      if result != 0:
         errors.append("Error: unable to check whether device is mounted.\n")
         errors.append("Command was [%s].\n" % mount_cmd)
         raise CedarBackupError()

      for line in lines:
         if mount_pattern.search(line):
            errors.append("Error: device appears to be mounted, which will likely cause a write failure.\n")
            raise CedarBackupError()
      
      device_dict['device']      = device

     
      ############################
      # Fill in device properties
      ############################

      # The properties command gives us a LOT of information back.  We don't
      # care about most of it.  We'll grab a few pieces of "identifying"
      # information, and then we'll also mark whether the drive is capable
      # of reading/writing multi-session discs, and whether it can be 
      # ejected.

      # Initialize for the search
      line  = ""
      lines = []

      description = ""
      eject       = False
      tray        = False
      multi       = False

      id_pattern     = re.compile(r"(^.*Identifikation.*:[ \t]*)('.*')(.*$)")
      eject_pattern  = re.compile(r"^.*Does support ejection.*$")
      tray_pattern   = re.compile(r"^.*Loading mechanism type: tray.*$")
      multi_pattern  = re.compile(r"^.*Does read multi-session.*$")

      device_dict['description'] = ""
      device_dict['multi']       = False
      device_dict['eject']       = False

      # Get the properties
      properties_cmd = "%s -prcap -dev %d,%d,%d" % (CDRECORD_PROGRAM, device_dict['devicename'], 
                                                    device_dict['scsibus'], device_dict['lun'])
      (result, lines) = execute_command(properties_cmd)
      if result != 0:
         errors.append("Error: unable to check properties on device.  Effective user may be improper.\n")
         errors.append("Command was [%s].\n" % properties_cmd)
         raise CedarBackupError()

      # Look for the particular entries we care about.
      for line in lines:
         parsed = id_pattern.search(line)
         if parsed:
            description = parsed.group(2)

         parsed = eject_pattern.search(line)
         if parsed:
            eject = True

         parsed = tray_pattern.search(line)
         if parsed:
            tray = True

         parsed = multi_pattern.search(line)
         if parsed:
            multi = True

      # Issue warnings as needed
      if not eject:
         warnings.append("Warning: device apparently does not support the eject operation.\n")

      if not tray:
         warnings.append("Warning: device apparently does not have a tray.  Eject operation would be unsafe.\n")

      if not multi:
         warnings.append("Warning: device apparently does not support multi-session discs.  Functionality disabled.\n")

      # Fill in the rest of the device dictionary
      device_dict['description'] = description

      if eject and tray:
         device_dict['eject'] = True

      if multi:
         device_dict['multi'] = True


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['device'] = device_dict

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


######################
# capacity() function
######################

def capacity(device_dict, new=False):

   ######################
   # Pydoc documentation
   ######################

   """
   Gather information about the capacity available on the current disc.

   Arguments:

      - **device_dict** : A 'device' dictionary returned from 'cdr.initialize()'

      - **new** : Indicates whether the disc should be treated as "new"

   Returns a dictionary of results:

      - 'used' : Space used on disc

      - 'available' : Space available on disc

      - 'boundaries' : Session disc boundaries as needed by cdrecord(1)

   Upon complete failure, None will be returned rather than a dictionary.

   Space used includes the required media lead-in (unless the disk is
   unused); space available attempts to provide a picture of how many
   bytes are available for data storage, including any required lead-in.
   
   The space values are returned as a dictionary containing the size in
   bytes, kbytes, 2-kb blocks and megabytes (keys 'bytes', 'kbytes',
   'blocks', 'mbytes').  The boundaries values are returned as a list 
   '[last-session-start, next-session-start]' and are in the form returned
   by cdrecord(1) (which is supposed to be compatible with mkisofs).
   
   If the disc is not already a multi-session disc, or if the passed-in
   new flag is True, the boundaries value will be None.
   """

   #######################
   # Initialize variables
   #######################

   results    = None

   boundaries = None
   used       = { }
   available  = { }

   media      = MEDIA_ATTRIBUTES[device_dict['media_type']]


   #########################################
   # Try to get the boundaries for the disc
   #########################################
   # We only bother getting the boundaries if this is not a new disc.

   if new:
      boundaries = None

   else:
      boundary_pattern = re.compile(r"(^)([0-9]*)(,)([0-9]*)($)")
      boundary_cmd = "%s -msinfo -dev=%d,%d,%d" % (CDRECORD_PROGRAM, device_dict['devicename'],
                                                   device_dict['scsibus'], device_dict['lun'])
      (result, lines) = execute_command(boundary_cmd)
      if result != 0:
         return None

      parsed = boundary_pattern.search(lines[0])
      if parsed:
         boundaries = [ int(parsed.group(2)), int(parsed.group(4)) ]


   #######################
   # Calculate capacities
   #######################

   # If we didn't find any boundaries (or if the next-session-start
   # value is <= 0), then we have to assume that the disc is unused as
   # of yet.  In this case, we'll just say that the available capacity
   # is the theoretical capacity for the media minus the initial
   # lead-in.
   #
   # If we did find boundaries, then we need to do some calculations.
   # Boundaries are given in blocks.  We really care about the "next
   # session start" value, which is essentially the number of blocks
   # that have already been used on the disc.  The remaining capacity is
   # then what is left available from the theoretical capacity minus the
   # standard lead-in for non-initial sessions.

   if boundaries is None or boundaries[1] == 0:

      used      =  { 'bytes'  : 0,
                     'kbytes' : 0,
                     'blocks' : 0,
                     'mbytes' : 0 }

      available =  { 'bytes'  : media['capacity']['bytes'] - (media['initial_lead_in'] * BYTES_PER_SECTOR),
                     'kbytes' : media['capacity']['kbytes'] - (media['initial_lead_in'] * KBYTES_PER_SECTOR),
                     'blocks' : media['capacity']['blocks'] - media['initial_lead_in'],
                     'mbytes' : media['capacity']['mbytes'] - (media['initial_lead_in'] * MBYTES_PER_SECTOR) }

   else:

      # Note: a sector and a block are equivalent here.  The boundaries 
      # and the media lead in are all calculated in blocks/sectors.

      blocks_available = media['capacity']['blocks'] - boundaries[1] - media['lead_in']
      if blocks_available < 0:
         blocks_available = 0

      used      =  { 'bytes'  : boundaries[1] * BYTES_PER_SECTOR,
                     'kbytes' : boundaries[1] * KBYTES_PER_SECTOR,
                     'blocks' : boundaries[1],
                     'mbytes' : boundaries[1] * MBYTES_PER_SECTOR }

      available =  { 'bytes'  : blocks_available * BYTES_PER_SECTOR,
                     'kbytes' : blocks_available * KBYTES_PER_SECTOR,
                     'blocks' : blocks_available,
                     'mbytes' : blocks_available * MBYTES_PER_SECTOR }


   #####################
   # Return the results
   #####################

   results               = { }
   results['used']       = used
   results['available']  = available
   results['boundaries'] = boundaries

   return results


##########################
# create_image() function
##########################

def create_image(iso_image, device_dict, dir_list, boundaries=[]):

   ######################
   # Pydoc documentation
   ######################

   """
   Create an ISO image based on a source directory and session boundary information.

   Arguments:
      
      - **iso_image** : Absolute path of the ISO image to be created

      - **device_dict** : A 'device' dictionary returned from 'cdr.initialize()'

      - **dir_list** : List of directories to be added to the image

      - **boundaries** : Disc boundaries as returned from 'cdr.capacity()'

   Returns a dictionary of results:

      - 'errors' : List of error messages or None

      - 'warnings' : List of warning messages or None

      - 'output' : Output from external command(s), as a list of strings

   If the 'errors' entry is None, the call was successful.

   If a boundaries value is passed in, then the function will merge to
   an existing image on disc.  Otherwise, the ISO image will be created
   as if it will be the first image written to disc.
   
   The elements in the dir_list list should themselves be dictionaries
   with two elements, 'prefix' and 'source_dir', where 'source_dir' is 
   the actual directory and 'prefix' is where the directory should appear
   in the ISO image (see the manpage for mkisofs(8) for more information
   on this concept).  There is no support for directory exclusion like
   mkisofs(8) allows from the command line.
   """

   #######################
   # Initialize variables
   #######################

   results             =  { }
   results['output']   = None
   results['errors']   = None
   results['warnings'] = None

   output              = []
   errors              = []
   warnings            = []


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      #################
      # Validate input
      #################

      # Trailing '/' characters will slightly mess up the resulting tree
      for dir in dir_list:
         if dir['prefix'][len(dir['prefix'])-1] == '/':
            dir['prefix'] = dir['prefix'][0:-1]
         if dir['source_dir'][len(dir['source_dir'])-1] == '/':
            dir['source_dir'] = dir['source_dir'][0:-1]


      ####################
      # Build the command
      ####################

      dir_string = ""
      for dir in dir_list:
         dir_string += "%s/=%s/ " % (dir['prefix'], dir['source_dir'])

      if boundaries:
         mkisofs_cmd = "%s -graft-points -r -o %s -C %d,%d -M %s %s" % (
                        MKISOFS_PROGRAM, iso_image, boundaries[0], boundaries[1], 
                        device_dict['device'], dir_string)
      else:
         mkisofs_cmd = "%s -graft-points -r -o %s %s" % (
                       MKISOFS_PROGRAM, iso_image, dir_string)


      ######################
      # Execute the command
      ######################

      (result, lines) = execute_command(mkisofs_cmd)
      output += lines
      if result != 0:
         errors.append("Error creating image.  See output for more details.\n")
         errors.append("Command was [%s].\n" % mkisofs_cmd)
         raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors
   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


#########################
# write_image() function
#########################

def write_image(iso_image, device_dict, new=False, multi=True):

   ######################
   # Pydoc documentation
   ######################

   """
   Writes an ISO image to a disc in the specified device.

   Arguments:
      
      - **iso_image** : Absolute path of the ISO image to be written

      - **device_dict** : A 'device' dictionary returned from 'cdr.initialize()'

      - **new** : Indicates whether disc should be treated as "new"

      - **multi** : Indicates whether multisession should be written

   Returns a dictionary of results:

      - 'errors' : List of error messages or None

      - 'warnings' : List of warning messages or None

      - 'output' : Output from external command(s), as a list of strings

   If the 'errors' entry is None, the call was successful.

   If the new argument is True, and the media type is rewritable, the
   disc will be blanked before writing it.  If the multi argument
   is True or is not given, and the device is capable of reading
   and writing multi-session discs, the disc will be written as
   multi-session.
   """

   #######################
   # Initialize variables
   #######################

   results              =  { }
   results['output']    = None
   results['errors']    = None
   results['warnings']  = None

   output               = []
   errors               = []
   warnings             = []


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      try:


         ############################################
         # Blank the disk, if needed and if possible
         ############################################

         # We need to blank the disc *before* trying to write to it.
         # I've seen cases where a disc has been hosed, and any attempt
         # to write to it (even with blank=fast as an option) just says
         # "Sorry, you're trying to write a disc that's >= 90 minutes,
         # this will fail."  The only way I found to fix it was to blank
         # the disc ahead of time.  Then, the write worked.
         #
         # We're building a command something like this:
         #
         #     cdrecord -v blank=fast speed=4 dev=0,0,0
         #
         # assuming that the media is rewritable.

         if new and MEDIA_ATTRIBUTES[device_dict['media_type']]['writable']:

            blank_cmd  = "%s -v" % CDRECORD_PROGRAM
            blank_cmd += " blank=fast"
            blank_cmd += " speed=%s" % (device_dict['drive_speed'])
            blank_cmd += " dev=%d,%d,%d" % (device_dict['devicename'], device_dict['scsibus'], device_dict['lun'])

            _eject(device_dict)

            (result, lines) = execute_command(blank_cmd)
            output += lines
            if result != 0:
               errors.append("Error blanking disc.  See results for more details.\n")
               errors.append("Command was [%s].\n" % blank_cmd)
               raise CedarBackupError()


         #################
         # Write the disc
         #################

         # We're building something like:
         #
         #     cdrecord -v speed=4 dev=0,0,0 -multi -data file.iso
         #
         # where the -multi argument is optional.

         write_cmd  = "%s -v" % CDRECORD_PROGRAM

         write_cmd += " speed=%s" % (device_dict['drive_speed'])
         write_cmd += " dev=%d,%d,%d" % (device_dict['devicename'], device_dict['scsibus'], device_dict['lun'])

         if multi and device_dict['multi']:
            write_cmd += " -multi"

         write_cmd += " -data"

         write_cmd += " %s" % iso_image

         _eject(device_dict)

         (result, lines) = execute_command(write_cmd)
         output += lines
         if result != 0:
            errors.append("Error writing to disc.  See results for more details.\n")
            errors.append("Command was [%s].\n" % write_cmd)
            raise CedarBackupError()


      #####################################
      # Handle all Cedar Backup exceptions
      #####################################

      except CedarBackupError:
         pass


   ###########
   # Clean up
   ###########
   # This done in a finally statement so we can be sure it happens. 

   finally:

      _eject(device_dict)


   #####################
   # Return the results
   #####################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors
   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


#######################################################################
# Private functions
#######################################################################

###################
# _eject() function
###################

def _eject(device_dict):

   """
   If the passed-in device is a capable of it, opens and closes the device's tray.
   """

   if device_dict['eject']:
      try:
         execute_command("%s %s" % (EJECT_PROGRAM, device_dict['device']))
         execute_command("%s -t %s" % (EJECT_PROGRAM, device_dict['device']))
      except:
         pass     # What can we do if it fails?
   return



########################################################################
# Module entry point
########################################################################

# Ensures that the module isn't run if someone just imports it.
if __name__ == '__main__':
   print "Module just exists to be imported, sorry."


