# -*- coding: iso-8859-1 -*-
# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
#              C E D A R
#          S O L U T I O N S       "Software done right."
#           S O F T W A R E
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Copyright (c) 2002-2003 Kenneth J. Pronovici.
# All rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# Version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Copies of the GNU General Public License are available from
# the Free Software Foundation website, http://www.gnu.org/.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.2)
# Project  : Cedar Backup
# Revision : $Id: config.py,v 1.8 2002/09/20 01:41:38 pronovic Exp $
# Purpose  : Provides functionality to read/parse config files
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

# This file was created with a width of 132 characters, and NO tabs.

########
# Notes
########

# This file is not intended to be an executable script.  Instead, it
# is a Python module file that provides common functionality for Cedar
# Backup scripts.  Import this file to use the functionality.


######################
# Pydoc documentation
######################

"""
Provides functionality to read/parse configuration files.

Functions that start with _ should be considered private to this
module, and should not be used by code outside this module.
"""

__author__  = "Kenneth J. Pronovici"


########################################################################
# Imported modules
########################################################################

# System modules
import os
import re
import string
import exceptions

# XML modules
from xml.dom.ext.reader import PyExpat
from xml.xpath import Evaluate
from xml.parsers.expat import ExpatError

# Cedar Backup modules
from CedarBackup.exceptions import CedarBackupError
from CedarBackup.filesystem import execute_command


#######################################################################
# Module-wide configuration and constants
#######################################################################

GROUP_PROGRAM = 'groups'


#######################################################################
# Public functions
#######################################################################

#########################
# read_config() function
#########################

def read_config(config_file, collect=False, stage=False, store=False, purge=False, rebuild=False):

   ######################
   # Pydoc documentation
   ######################

   """
   Read as many configuration sections as possible out of the config file.

   Arguments:

      - **config_file** : Path to the configuration file

      - **(others)** : Indicate which action(s) will be taken

   Returns a dictionary of results:

      - 'config' : Dictionary describing the read-in configuration

      - 'warnings' : List of warnings about the configuration or None

      - 'errors' : List of errors with the configuration, or None

   If there are any errors at all, the 'config' entry will be None.
   
   The reference and global options sections are always required along
   with at least one of the collect, stage, store or purge configuration
   sections.  It's allowable for a single configuration file to contain
   any or all of the four action-specific sections, however.

   This function is mostly implemented in terms of other private, 
   more specialized functions.
   """

   ############
   # Variables
   ############
   
   results              = { }
   results['config']    = None
   results['errors']    = None
   results['warnings']  = None
   
   config               = { }
   errors               = [ ]
   warnings             = [ ]


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ##########################
      # Set up the XML DOM tree
      ##########################

      try:
         dom = PyExpat.Reader().fromString(open(config_file, "r").read())
      except IOError:
         errors.append("Error: unable to read configuration file '%s'.\n" % config_file)
         raise CedarBackupError()
      except ExpatError:
         errors.append("Error: unable to parse configuration file '%s'.\n" % config_file)
         raise CedarBackupError()


      ######################################################
      # Read each of the individual configurations sections
      ######################################################

      # Read the reference section
      call = _read_reference(dom)
      if call is None:
         errors.append("Required reference section is missing.\n")
         raise CedarBackupError()
      else:
         if call['warnings'] is not None:
            warnings += call['warnings']
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()
         config['reference'] = call['dict']

      # Read the options section
      call = _read_options(dom)
      if call is None:
         errors.append("Required options section is missing.\n")
         raise CedarBackupError()
      else:
         if call['warnings'] is not None:
            warnings += call['warnings']
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()
         config['options'] = call['dict']
   
      # Read the collect section
      call = _read_collect(dom)
      if call is None:
         call['collect'] = None
      else:
         if call['warnings'] is not None:
            warnings += call['warnings']
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()
         config['collect'] = call['dict']

      # Read the stage section
      call = _read_stage(dom)
      if call is None:
         config['stage'] = None
      else:
         if call['warnings'] is not None:
            warnings += call['warnings']
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()
         config['stage'] = call['dict']

      # Read the store section
      call = _read_store(dom)
      if call is None:
         config['store'] = None
      else:
         if call['warnings'] is not None:
            warnings += call['warnings']
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()
         config['store'] = call['dict']
   
      # Read the purge section
      call = _read_purge(dom)
      if call is None:
         config['purge'] = None
      else:
         if call['warnings'] is not None:
            warnings += call['warnings']
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()
         config['purge'] = call['dict']
   
      # Check to make sure we have at least one configuration section
      if config['collect'] is None and config['stage'] is None and config['store'] is None and config['purge'] is None:
         errors.append("At least one section (collect, stage, store, purge) is required.\n")
         raise CedarBackupError()

      # Check to make sure the config pieces for the individual actions exist.
      if collect and config['collect'] is None:
         errors.append("Required collect section is missing.\n")
         raise CedarBackupError()
   
      if stage and config['stage'] is None:
         errors.append("Required stage section is missing.\n")
         raise CedarBackupError()

      if store and config['store'] is None:
         errors.append("Required store section is missing.\n")
         raise CedarBackupError()

      if purge and config['purge'] is None:
         errors.append("Required purge section is missing.\n")
         raise CedarBackupError()

      if rebuild and config['store'] is None:
         errors.append("Required store section is missing.\n")
         raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['config'] = config

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


#######################################################################
# Private functions
#######################################################################

#############################
# _read_reference() function
#############################

# Configuration items:

#  Variable                      XPath Expression                          Description
#  -----------------------------------------------------------------------------------------------------------------------------
#  dict['author']                author                                    Author of config file.
#  dict['revision']              revision                                  Revision of config file.
#  dict['description']           description                               Description for config file.  
#           * All XPath Expressions start with //cb_config/reference/

# No validations are done on these configuration items.

def _read_reference(dom):

   ######################
   # Pydoc documentation
   ######################

   """
   Read the reference configuration section.

   Arguments:

      - **dom** : XML DOM tree representation of config file

   Returns a dictionary of results:

      - 'dict' : A dictionary of configuration values
      - 'errors' : List of error messages or None
      - 'warnings' : List of warning messages or None

   If the 'errors' entry is None, the call was successful.  If
   the section does not exist in the document, None is returned.
   """

   ############
   # Variables
   ############

   results             = { }
   results['dict']     = None
   results['errors']   = None
   results['warnings'] = None

   dictx               = { }
   errors              = [ ]
   warnings            = [ ]


   #################################
   # See whether the section exists
   #################################

   try:
      if Evaluate('//cb_config/reference', dom.documentElement) == []:
         return None
   except:
      return None


   ############################
   # Read values from the file
   ############################

   dictx['author']        = Evaluate('string(//cb_config/reference/author)', dom.documentElement)
   dictx['revision']      = Evaluate('string(//cb_config/reference/revision)', dom.documentElement)
   dictx['description']   = Evaluate('string(//cb_config/reference/description)', dom.documentElement)


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['dict'] = dictx

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


###########################
# _read_options() function
###########################

# Configuration items:

#  Variable                      XPath Expression                          Description
#  -----------------------------------------------------------------------------------------------------------------------------
#  dict['starting_day']          starting_day                              Day that a week starts (using full day names)
#  dict['working_dir']           working_dir                               Directory that the script can use for temp files, etc.
#  dict['backup_user']           backup_user                               Local backup user.
#  dict['backup_group']          backup_group                              Local backup group.
#  dict['rcp_command']           rcp_command                               Command (including options) to use for remote copy.
#           * All XPath Expressions start with //cb_config/options/

# Validation is done on most of the configuration items:
#
#     o starting day must be a valid day of the week ("monday", "tuesday", etc.)
#     o working directory must be specified as an absolute path and must be writable
#     o backup user must exist and must have backup group in its groupset
#     o command portion of rcp value (i.e. '/usr/bin/scp' in value '/usr/bin/scp -1') must be executable

def _read_options(dom):

   ######################
   # Pydoc documentation
   ######################

   """
   Read the options configuration section.

   Arguments:

      - **dom** : XML DOM tree representation of config file

   Returns a dictionary of results:

      - 'dict' : A dictionary of configuration values
      - 'errors' : List of error messages or None
      - 'warnings' : List of warning messages or None

   If the 'errors' entry is None, the call was successful.  If
   the section does not exist in the document, None is returned.
   """

   ############
   # Variables
   ############

   results             = { }
   results['dict']     = None
   results['errors']   = None
   results['warnings'] = None

   dictx               = { }
   errors              = [ ]
   warnings            = [ ]


   #################################
   # See whether the section exists
   #################################

   try:
      if Evaluate('//cb_config/options', dom.documentElement) == []:
         return None
   except:
      return None


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ############################
      # Read values from the file
      ############################

      dictx['starting_day']  = Evaluate('string(//cb_config/options/starting_day)', dom.documentElement)
      dictx['working_dir']   = Evaluate('string(//cb_config/options/working_dir)', dom.documentElement)
      dictx['backup_user']   = Evaluate('string(//cb_config/options/backup_user)', dom.documentElement)
      dictx['backup_group']  = Evaluate('string(//cb_config/options/backup_group)', dom.documentElement)
      dictx['rcp_command']   = Evaluate('string(//cb_config/options/rcp_command)', dom.documentElement)


      #############################
      # Validate the configuration
      #############################

      if dictx['starting_day'] not in ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday' ]:
         errors.append("Error: starting day '%s' invalid; must be valid weekday name (i.e. 'sunday', etc.)\n" 
                        % dictx['starting_day'])
         raise CedarBackupError()

      if not os.path.isabs(dictx['working_dir']):
         errors.append("Error: working directory '%s' invalid; must be specified as an absolute path.\n" 
                        % dictx['working_dir'])
         raise CedarBackupError()

      if not os.path.isdir(dictx['working_dir']) or not os.access(dictx['working_dir'], os.W_OK):
         errors.append("Error: working directory '%s' invalid; does not appear be writable by effective user.\n"
                        % dictx['working_dir'])
         raise CedarBackupError()

      id_command = "%s %s" % (GROUP_PROGRAM, dictx['backup_user'])
      (result, lines) = execute_command(id_command)
      if result != 0:
         errors.append("Error: backup user '%s' invalid; does not appear to exist.\n" % dictx['backup_user'])
         raise CedarBackupError()

      if dictx['backup_group'] not in string.split(lines[0][string.find(lines[0], ":")+1:]):
         errors.append("Error: backup group '%s' invalid; not in '%s' user's groupset.\n" 
                       % (dictx['backup_group'],dictx['backup_user']))
         raise CedarBackupError()
      
      program = string.split(dictx['rcp_command'])[0]
      if not os.path.isabs(program):
         errors.append("Error: invalid rcp command '%s'; must be specified as an absolute path.\n" % program)
         raise CedarBackupError()
      if not os.path.isfile(program) or not os.access(program, os.X_OK):
         errors.append("Error: invalid rcp command '%s'; apparently not available.\n" % program)
         raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['dict'] = dictx

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


###########################
# _read_collect() function
###########################

# Configuration items:

#  Variable                                  XPath Expression              Description
#  -----------------------------------------------------------------------------------------------------------------------------
#  dict['collect_dir']                       collect_dir                   Directory into which tarfiles are collected.
#  dict['archive_mode']                      archive_mode                  Archive mode.  One of [tar, targz, tarbz2, tarz].
#  dict['ignore_file']                       ignore_file                   File that tells us to ignore a directory.
#  dict['dir'][i]['abs_path']                dir[x]/abs_path               Name of directory to be archived (absolute path).
#  dict['dir'][i]['mode']                    dir[x]/mode                   Archive mode for dir.  One of [daily, weekly, incr].
#  dict['dir'][i]['exclude'][j]              dir[x]/exclude[y]/abs_path    Name of excluded file or directory (absolute path).
#           * All XPath Expressions start with //cb_config/collect/

# Validation is done on most of the configuration items:
#
#     o archive mode must be one of [tar, targz, tarbz2, tarz]
#     o paths must be absolute 
#     o the collect directory must be writable
#     o mode must be one of [daily, weekly]

def _read_collect(dom):

   ######################
   # Pydoc documentation
   ######################

   """
   Read the collect configuration section.

   Arguments:

      - **dom** : XML DOM tree representation of config file

   Returns a dictionary of results:

      - 'dict' : A dictionary of configuration values
      - 'errors' : List of error messages or None
      - 'warnings' : List of warning messages or None

   If the 'errors' entry is None, the call was successful.  If
   the section does not exist in the document, None is returned.
   """

   ############
   # Variables
   ############

   results             = { }
   results['dict']     = None
   results['errors']   = None
   results['warnings'] = None

   dictx               = { }
   errors              = [ ]
   warnings            = [ ]


   #################################
   # See whether the section exists
   #################################

   try:
      if Evaluate('//cb_config/collect', dom.documentElement) == []:
         return None
   except:
      return None


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ############################
      # Read values from the file
      ############################

      dictx['collect_dir']  = Evaluate('string(//cb_config/collect/collect_dir)', dom.documentElement)
      dictx['archive_mode'] = Evaluate('string(//cb_config/collect/archive_mode)', dom.documentElement)
      dictx['ignore_file'] = Evaluate('string(//cb_config/collect/ignore_file)', dom.documentElement)

      i = 0
      dictx['dir'] = [ ]
      while Evaluate('//cb_config/collect/dir[%d]' % (i+1), dom.documentElement) != []:
         dictx['dir'].append({ })
         dictx['dir'][i]['abs_path'] = Evaluate('string(//cb_config/collect/dir[%d]/abs_path)' % (i+1), dom.documentElement)
         dictx['dir'][i]['mode'] = Evaluate('string(//cb_config/collect/dir[%d]/mode)' % (i+1), dom.documentElement)

         j = 0
         dictx['dir'][i]['exclude'] = [ ]
         while Evaluate('//cb_config/collect/dir[%d]/exclude/abs_path[%d]' % (i+1, j+1), dom.documentElement) != []:
            dictx['dir'][i]['exclude'].append(Evaluate('string(//cb_config/collect/dir[%d]/exclude/abs_path[%d])' % 
                                             (i+1, j+1), dom.documentElement))
            j += 1

         i += 1


      #############################
      # Validate the configuration
      #############################

      if not os.path.isabs(dictx['collect_dir']):
         errors.append("Error: collect directory '%s' invalid; must be specified as an absolute path.\n" % dictx['collect_dir'])
         raise CedarBackupError()

      if not os.path.isdir(dictx['collect_dir']) or not os.access(dictx['collect_dir'], os.W_OK):
         errors.append("Error: collect directory '%s' invalid; does not appear be writable by effective user.\n" 
                        % dictx['collect_dir'])
         raise CedarBackupError()

      if dictx['archive_mode'] not in ['tar', 'targz', 'tarbz2', 'tarz']:
         errors.append("Error: archive mode '%s' invalid; must be one of [tar, targz, tarbz2, tarz]\n" 
                        % dictx['archive_mode'])
         raise CedarBackupError()

      for d in dictx['dir']:
         if not os.path.isabs(d['abs_path']):
            errors.append("Error: directory path '%s' invalid; must be specified as an absolute path.\n" % d['abs_path'])
            raise CedarBackupError()
         if d['mode'] not in ['daily', 'weekly', 'incr']:
            errors.append("Error: archive mode '%s' invalid; must be one of [daily, weekly, incr]\n" % d['mode'])
            raise CedarBackupError()

         for path in d['exclude']:
            if not os.path.isabs(path):
               errors.append("Error: exclude path '%s' invalid; must be specified as an absolute path.\n" % path)
               raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['dict'] = dictx

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


#########################
# _read_stage() function
#########################

# Configuration items:

#  Variable                                  XPath Expression              Description
#  -----------------------------------------------------------------------------------------------------------------------------
#  dict['staging_dir']                       staging_dir                   Directory in which to stage backups from peer machines.
#  dict['peer'][i]['name']                   peer[x]/name                  Peer name.
#  dict['peer'][i]['type']                   peer[x]/type                  Peer type.  One of [remote, local].
#  dict['peer'][i]['collect_dir']            peer[x]/collect_dir           Directory on peer machine where backups are collected.
#  dict['peer'][i]['backup_user']            peer[x]/backup_user           Remote backup user, if peer type is remote. 
#           * All XPath Expressions start with //cb_config/stage/

# Validation is done on most of the configuration items:
#
#     o paths must be absolute 
#     o type must be one of [remote, local]
#     o remote peers must list a backup user

def _read_stage(dom):

   ######################
   # Pydoc documentation
   ######################

   """
   Read the stage configuration section.

   Arguments:

      - **dom** : XML DOM tree representation of config file

   Returns a dictionary of results:

      - 'dict' : A dictionary of configuration values
      - 'errors' : List of error messages or None
      - 'warnings' : List of warning messages or None

   If the 'errors' entry is None, the call was successful.  If
   the section does not exist in the document, None is returned.
   """

   ############
   # Variables
   ############

   results             = { }
   results['dict']     = None
   results['errors']   = None
   results['warnings'] = None

   dictx               = { }
   errors              = [ ]
   warnings            = [ ]


   #################################
   # See whether the section exists
   #################################

   try:
      if Evaluate('//cb_config/stage', dom.documentElement) == []:
         return None
   except:
      return None


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ############################
      # Read values from the file
      ############################

      dictx['staging_dir']  = Evaluate('string(//cb_config/stage/staging_dir)', dom.documentElement)

      i = 0
      dictx['peer'] = [ ]
      while Evaluate('//cb_config/stage/peer[%d]' % (i+1), dom.documentElement) != []:
         dictx['peer'].append({ })
         dictx['peer'][i]['name'] = Evaluate('string(//cb_config/stage/peer[%d]/name)' % (i+1), dom.documentElement)
         dictx['peer'][i]['type'] = Evaluate('string(//cb_config/stage/peer[%d]/type)' % (i+1), dom.documentElement)
         dictx['peer'][i]['collect_dir'] = Evaluate('string(//cb_config/stage/peer[%d]/collect_dir)' % (i+1), dom.documentElement)
         dictx['peer'][i]['backup_user'] = Evaluate('string(//cb_config/stage/peer[%d]/backup_user)' % (i+1), dom.documentElement)
         i += 1


      #############################
      # Validate the configuration
      #############################

      if not os.path.isabs(dictx['staging_dir']):
         errors.append("Error: staging directory '%s' invalid; must be specified as an absolute path.\n" % dictx['staging_dir'])
         raise CedarBackupError()

      if not os.path.isdir(dictx['staging_dir']) or not os.access(dictx['staging_dir'], os.W_OK):
         errors.append("Error: staging directory '%s' invalid; does not appear be writable by effective user.\n" 
                        % dictx['staging_dir'])
         raise CedarBackupError()

      for peer in dictx['peer']:
         if peer['type'] not in ['local', 'remote']:
            errors.append("Error: peer type '%s' invalid; must be one of [local, remote].\n" % peer['type'])
            raise CedarBackupError()
         if not os.path.isabs(peer['collect_dir']):
            errors.append("Error: peer collect directory '%s' invalid; must be specified as an absolute path.\n" 
                           % peer['collect_dir'])
            raise CedarBackupError()
         if peer['type'] == 'remote' and peer['backup_user'] == "":
            errors.append("Error: backup user is required for all remote peers.\n")
            raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['dict'] = dictx

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


#########################
# _read_store() function
#########################

# Configuration items:

#  Variable                                  XPath Expression              Description
#  -----------------------------------------------------------------------------------------------------------------------------
#  dict['source_dir']                        source_dir                    Directory from which to create the ISO image.
#  dict['target_device']                     target_device                 System backup device (i.e. /dev/cdrw).
#  dict['target_scsi_id']                    target_scsi_id                SCSI ID for device in form '[ATA:]scsibus,target,lun'
#  dict['drive_speed']                       drive_speed                   Drive speed (i.e. 2 for 2x CD-R, 4 for 4x CD-R, etc.).
#  dict['media_type']                        media_type                    Media type.  One of [cdrw-74, cdr-74].
#  dict['check_data']                        check_data                    Run consistency check?  Must be one of [Y,N].
#           * All XPath Expressions start with //cb_config/store/

# Validation is done on most of the configuration items:
#
#     o The device must be an absolute path, must exist, and must be writable by the effective user id
#     o The SCSI id must be in the form "a,b,c"
#     o The drive speed must be an integer >= 1

def _read_store(dom):

   ######################
   # Pydoc documentation
   ######################

   """
   Read the store configuration section.

   Arguments:

      - **dom** : XML DOM tree representation of config file

   Returns a dictionary of results:

      - 'dict' : A dictionary of configuration values
      - 'errors' : List of error messages or None
      - 'warnings' : List of warning messages or None

   If the 'errors' entry is None, the call was successful.  If
   the section does not exist in the document, None is returned.
   """

   ############
   # Variables
   ############

   results             = { }
   results['dict']     = None
   results['errors']   = None
   results['warnings'] = None

   dictx               = { }
   errors              = [ ]
   warnings            = [ ]


   #################################
   # See whether the section exists
   #################################

   try:
      if Evaluate('//cb_config/store', dom.documentElement) == []:
         return None
   except:
      return None


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ############################
      # Read values from the file
      ############################

      dictx['source_dir']      = Evaluate('string(//cb_config/store/source_dir)', dom.documentElement)
      dictx['target_device']   = Evaluate('string(//cb_config/store/target_device)', dom.documentElement)
      target_scsi_id          = Evaluate('string(//cb_config/store/target_scsi_id)', dom.documentElement)
      drive_speed             = Evaluate('string(//cb_config/store/drive_speed)', dom.documentElement)
      dictx['media_type']      = Evaluate('string(//cb_config/store/media_type)', dom.documentElement)
      temp_check_data         = string.upper(Evaluate('string(//cb_config/store/check_data)', dom.documentElement))


      #############################
      # Validate the configuration
      #############################

      if not os.path.isabs(dictx['source_dir']):
         errors.append("Error: source directory '%s' invalid; must be specified as an absolute path.\n" % dictx['source_dir'])
         raise CedarBackupError()

      if not os.path.isabs(dictx['target_device']):
         errors.append("Error: target device '%s' invalid; must be specified as an absolute path.\n" % dictx['target_device'])
         raise CedarBackupError()

      if os.path.islink(dictx['target_device']):
         warnings.append("Warning: target device '%s' is a link; some safeguards may not catch problems.\n" 
                         % dictx['target_device'])
         results['warnings'] = warnings

      if not os.access(dictx['target_device'], os.W_OK):
         errors.append("Error: target device '%s' invalid; does not appear be writable by effective user.\n" 
                        % dictx['target_device'])
         raise CedarBackupError()

      ata_scsi_id_pattern = re.compile(r"^\s*ATA:[0-9][0-9]*\s*,\s*[0-9][0-9]*\s*,\s*[0-9][0-9]*\s*$")
      scsi_id_pattern = re.compile(r"^\s*[0-9][0-9]*\s*,\s*[0-9][0-9]*\s*,\s*[0-9][0-9]*\s*$")
      if not ata_scsi_id_pattern.search(target_scsi_id) and not scsi_id_pattern.search(target_scsi_id):
         errors.append("Error: SCSI id '%s' invalid; must be in the form '[ATA:]scsibus,target,lun'.\n" % target_scsi_id)
         raise CedarBackupError()

      dictx['target_scsi_id'] = string.strip(target_scsi_id);

      try:
         dictx['drive_speed'] = int(drive_speed)
      except ValueError:
         errors.append("Error: drive speed '%s' invalid; must be an integer >= 1.\n" % drive_speed)
         raise CedarBackupError()

      if dictx['drive_speed'] < 1:
         errors.append("Error: drive speed '%d' invalid; must be an integer >= 1.\n" % dictx['drive_speed'])
         raise CedarBackupError()

      if dictx['media_type'] not in ['cdr-74', 'cdrw-74']:
         errors.append("Error: media type '%s' invalid; must be one of [cdr-74, cdrw-74].\n" % dictx['media_type'])
         raise CedarBackupError()

      if temp_check_data not in ['Y', 'N']:
         errors.append("Error: check-data Y/N value '%s' invalid.\n" % dictx['check_data'])
         raise CedarBackupError()

      if temp_check_data == "Y":
         dictx['check_data'] = True
      else:
         dictx['check_data'] = False
      

   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['dict'] = dictx

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


#########################
# _read_purge() function
#########################

# Configuration items:

#  Variable                                  XPath Expression              Description
#  -----------------------------------------------------------------------------------------------------------------------------
#  dict['dir'][i]['abs_path']                dir[x]/abs_path               Name of directory to be purged (absolute path).
#  dict['dir'][i]['retain_days']             dir[x]/retain_days            Number of days to retain files/dirs in the directory.

# Validation is done on most of the configuration items:
#
#     o The path must be absolute
#     o The retain days value must be >= 0

def _read_purge(dom):

   ######################
   # Pydoc documentation
   ######################

   """
   Read the purge configuration section.

   Arguments:

      - **dom** : XML DOM tree representation of config file

   Returns a dictionary of results:

      - 'dict' : A dictionary of configuration values
      - 'errors' : List of error messages or None
      - 'warnings' : List of warning messages or None

   If the 'errors' entry is None, the call was successful.  If
   the section does not exist in the document, None is returned.
   """

   ############
   # Variables
   ############

   results             = { }
   results['errors']   = None
   results['warnings'] = None

   dictx               = { }
   errors              = [ ]
   warnings            = [ ]

   retain_days         = [ ]


   #################################
   # See whether the section exists
   #################################

   try:
      if Evaluate('//cb_config/purge', dom.documentElement) == []:
         return None
   except:
      return None


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ############################
      # Read values from the file
      ############################

      i = 0
      dictx['dir'] = [ ]
      while Evaluate('//cb_config/purge/dir[%d]' % (i+1), dom.documentElement) != []:
         dictx['dir'].append({ })
         dictx['dir'][i]['abs_path'] = Evaluate('string(//cb_config/purge/dir[%d]/abs_path)' % (i+1), dom.documentElement)
         retain_days.append(Evaluate('string(//cb_config/purge/dir[%d]/retain_days)' % (i+1), dom.documentElement))
         i += 1


      #############################
      # Validate the configuration
      #############################

      i = 0
      for d in dictx['dir']:
         if not os.path.isabs(d['abs_path']):
            errors.append("Error: purge directory '%s' invalid; must be specified as an absolute path.\n" % d['abs_path'])
            raise CedarBackupError()

         try:
            d['retain_days'] = float(retain_days[i])
         except ValueError:
            errors.append("Error: retain days '%s' invalid; must be number >= 0.0.\n" % retain_days[i])
            raise CedarBackupError()

         if d['retain_days'] < 0:
            errors.append("Error: retain days '%f' invalid; must be number >= 0.0.\n" % d['retain_days'])
            raise CedarBackupError()

         i += 1


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #####################
   # Return the results
   #####################

   if len(errors) > 0:
      results['errors'] = errors
   else:
      results['dict'] = dictx

   if len(warnings) > 0:
      results['warnings'] = warnings

   return results


########################################################################
# Module entry point
########################################################################

# Ensures that the module isn't run if someone just imports it.
if __name__ == '__main__':
   print "Module just exists to be imported, sorry."


