# -*- coding: iso-8859-1 -*-
# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
#              C E D A R
#          S O L U T I O N S       "Software done right."
#           S O F T W A R E
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Copyright (c) 2002-2003 Kenneth J. Pronovici.
# All rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# Version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Copies of the GNU General Public License are available from
# the Free Software Foundation website, http://www.gnu.org/.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.2)
# Project  : Cedar Backup
# Revision : $Id: functional.py,v 1.12 2002/09/20 01:41:38 pronovic Exp $
# Purpose  : Provides high-level application functionality for the project
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

# This file was created with a width of 132 characters, and NO tabs.

########
# Notes
########

# This file is not intended to be an executable script.  Instead, it
# is a Python module file that provides common functionality for Cedar
# Backup scripts.  Import this file to use the functionality.


######################
# Pydoc documentation
######################

"""
Provides high-level functionality, in terms of other CedarBackup modules.

All of these functions return a dictionary that contains three
entries, 'output', 'errors', 'warnings'.  All three are lists of
strings suitable for being passed to a writelines()-style function.
The 'output' entry contains the output of any external commands run as
a result of calling the function.  The 'errors' entry contains error
messages describing any errors encountered while processing.  The
'warnings' entry contains any warnings generated while processing.

Functions that start with _ should be considered private to this
module, and should not be used by code outside this module.
"""

__author__  = "Kenneth J. Pronovici"


########################################################################
# Imported modules
########################################################################

# System modules
import os
import re
import string
import tempfile
import time
import exceptions
import pwd
import grp

# Cedar Backup modules
from CedarBackup.exceptions import CedarBackupError
import CedarBackup.filesystem as filesystem
import CedarBackup.cdr as cdr


#######################################################################
# Module-wide configuration and constants
#######################################################################

PREFIX_TIME_FORMAT   = "%Y/%m/%d"
DIR_TIME_FORMAT      = "%Y/%m/%d"

COLLECT_INDICATOR    = "cback.collect"
STAGE_INDICATOR      = "cback.stage"


#######################################################################
# Public functions
#######################################################################

#########################
# run_collect() function
#########################

def run_collect(config, full=False):

   ######################
   # Pydoc documentation
   ######################

   """
   Executes program "collect" functionality.

   Also accepts the 'full' argument, which indicates that all files
   and directories should be backed up, regardless of whether they are
   flagged as daily, weekly or incremental.  This allows for "do a full
   backup now" functionality.

   Note that it might be worthwhile to regularly purge the Cedar Backup
   working directory to get rid of old, unused incremental backup
   records.
   """

   ############
   # Variables
   ############

   results             = { }
   results['output']   = None
   results['errors']   = None
   results['warnings'] = None

   output              = [ ]
   errors              = [ ]
   warnings            = [ ]

   reset_digest        = ""


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ################################################
      # Check whether to reset the incremental digest
      ################################################
      # We want to reset the incremental digest on disk at the beginning of the
      # week, or whenever we do a full backup.

      if full or _today_is_start(config['options']['starting_day']):
         reset_digest = True
      else:
         reset_digest = False
   

      ##################################
      # Loop for each collect directory
      ##################################

      for d in config['collect']['dir']:

         #############################################
         # Decide whether to deal with this directory
         #############################################
         # We always do daily/incremental, but only do weekly at the start of a week.

         if (full or d['mode'] == 'daily' or d['mode'] == 'incr' or
              (d['mode'] == 'weekly' and _today_is_start(config['options']['starting_day'])) ):

            ##################
            # Build filenames
            ##################

            # Yup, we assume a UNIX-style path
            basename = d['abs_path']
            basename = re.sub("^\/", "", basename)
            basename = re.sub("\/", "-", basename)
            basename = re.sub("\s", "_", basename)

            if d['mode'] == "incr" and not full:
               digest_file = os.path.join(config['options']['working_dir'], "%s.sha" % basename)
            else:
               digest_file = None

            if config['collect']['archive_mode'] == 'tar':
               tarfile = os.path.join(config['collect']['collect_dir'], "%s.tar" % basename)
            if config['collect']['archive_mode'] == "targz":
               tarfile = os.path.join(config['collect']['collect_dir'], "%s.tar.gz" % basename)
            elif config['collect']['archive_mode'] == 'tarbz2':
               tarfile = os.path.join(config['collect']['collect_dir'], "%s.tar.bz2" % basename)
            elif config['collect']['archive_mode'] == 'tarz':
               tarfile = os.path.join(config['collect']['collect_dir'], "%s.tar.Z" % basename)


            #####################
            # Create the tarfile
            #####################

            call = filesystem.tar_tree(dir=d['abs_path'], 
                                       tarfile=tarfile, 
                                       working_dir=config['options']['working_dir'], 
                                       method=config['collect']['archive_mode'],
                                       digest_file=digest_file, 
                                       reset_digest=reset_digest,
                                       ignore_file=config['collect']['ignore_file'],
                                       exclude_list=d['exclude'],
                                       user=config['options']['backup_user'],
                                       group=config['options']['backup_group'])

            if call['output'] is not None:
               output += call['output']

            if call['warnings'] is not None:
               warnings += call['warnings']

            if call['errors'] is not None:
               errors += call['errors']
               raise CedarBackupError()


      ###########################
      # Write the indicator file
      ###########################

      call = filesystem.write_indicator(indicator=os.path.join(config['collect']['collect_dir'], COLLECT_INDICATOR),
                                        user=config['options']['backup_user'],
                                        group=config['options']['backup_group'])

      if call['errors'] is not None:
         errors += call['errors']
         raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #################
   # Return results
   #################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors;
   if len(warnings) > 0:
      results['warnings'] = warnings
      
   return results


#######################
# run_stage() function
#######################

def run_stage(config):

   ######################
   # Pydoc documentation
   ######################

   """
   Executes  program "stage" functionality.
   """

   ############
   # Variables
   ############

   results             = { }
   results['output']   = None
   results['errors']   = None
   results['warnings'] = None

   output              = [ ]
   errors              = [ ]
   warnings            = [ ]


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      #####################
      # Preliminary set-up
      #####################

      try:
         uid = pwd.getpwnam(config['options']['backup_user'])[2]
         gid = grp.getgrnam(config['options']['backup_group'])[2]
      except:
         errors.append("Error: backup user/group apparently invalid.\n")
         raise CedarBackupError()


      ####################################
      # Create the base staging directory
      ####################################

      daily_dir = os.path.join(config['stage']['staging_dir'], time.strftime(DIR_TIME_FORMAT))

      if os.path.exists(daily_dir) and not os.path.isdir(daily_dir):
         errors.append("Error: daily directory's name ('%s') already exists, but is not a directory.\n" % daily_dir)
         raise CedarBackupError()

      if not os.path.isdir(daily_dir):
         try:
            os.makedirs(daily_dir, 0750)
            os.chown(daily_dir, uid, gid)                   # staging/2002/05/23
            os.chown("%s/.." % daily_dir, uid, gid)         # staging/2002/05
            os.chown("%s/../.." % daily_dir, uid, gid)      # staging/2002
         except:
            errors.append("Error: unable to create daily staging directory within '%s' using Python built-ins.\n" % (
                          config['stage']['staging_dir']))
            raise CedarBackupError()


      ####################
      # Stage local files
      ####################
      # Loop for each local peer, check its indicator, and then copy its files

      for peer in config['stage']['peer']:

         # Only deal with local peers
         if peer['type'] != 'local':
            continue

         # Display a warning if the directory isn't complete
         if not filesystem.local_indicator_exists(os.path.join(peer['collect_dir'], COLLECT_INDICATOR)):
            warnings.append("Warning: directory %s:%s was not ready to be staged.\n" % 
                            (peer['name'], os.path.join(peer['collect_dir'])) )
            continue
         
         # Create the staging directory for this peer
         peer_dir = os.path.join(daily_dir, peer['name'])

         if os.path.exists(peer_dir) and not os.path.isdir(peer_dir):
            errors.append("Error: peer directory's name ('%s') already exists, but is not a directory.\n" % peer_dir)
            raise CedarBackupError()
   
         if not os.path.isdir(peer_dir):
            try:
               os.makedirs(peer_dir)
               os.chown(peer_dir, uid, gid)
            except:
               errors.append("Error: unable to create directory '%s' using Python built-ins.\n" % peer_dir)
               raise CedarBackupError()
               
         # Copy the peer collect directory into the staging directory
         call = filesystem.copy_local_dir(sdir=peer['collect_dir'], 
                                          ddir=peer_dir, 
                                          user=config['options']['backup_user'], 
                                          group=config['options']['backup_group'])
           
         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()


      #####################
      # Stage remote files
      #####################
      # Loop for each remote peer, check its indicator, and then copy its files

      for peer in config['stage']['peer']:

         # Only deal with local peers
         if peer['type'] != 'remote':
            continue

         # Display a warning if the directory isn't complete
         if not filesystem.remote_indicator_exists(rcp=config['options']['rcp_command'],
                                                   rhost=peer['name'],
                                                   working_dir=config['options']['working_dir'],
                                                   indicator=os.path.join(peer['collect_dir'], COLLECT_INDICATOR),
                                                   luser=config['options']['backup_user'],
                                                   ruser=peer['backup_user']):
            warnings.append("Warning: directory %s:%s was not ready to be staged.\n" 
                            % (peer['name'], os.path.join(peer['collect_dir'])))
            continue
         
         # Create the staging directory for this peer
         peer_dir = os.path.join(daily_dir, peer['name'])

         if os.path.exists(peer_dir) and not os.path.isdir(peer_dir):
            errors.append("Error: peer directory's name ('%s') already exists, but is not a directory.\n" % peer_dir)
            raise CedarBackupError()
   
         if not os.path.isdir(peer_dir):
            try:
               os.makedirs(peer_dir)
               os.chown(peer_dir, uid, gid)
            except:
               errors.append("Error: unable to create directory '%s'.\n" % peer_dir)
               raise CedarBackupError()
               
         # Copy the peer collect directory into the staging directory
         call = filesystem.copy_remote_dir(rcp=config['options']['rcp_command'],
                                           rhost=peer['name'],
                                           sdir=peer['collect_dir'], 
                                           ddir=peer_dir, 
                                           luser=config['options']['backup_user'],
                                           ruser=peer['backup_user'])
         if call['output'] is not None:
            output += call['output']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()


      ###########################
      # Write the indicator file
      ###########################

      call = filesystem.write_indicator(user=config['options']['backup_user'],
                                        group=config['options']['backup_group'],
                                        indicator=os.path.join(daily_dir, STAGE_INDICATOR))

      if call['errors'] is not None:
         errors += call['errors']
         raise CedarBackupError()


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #################
   # Return results
   #################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors;
   if len(warnings) > 0:
      results['warnings'] = warnings
      
   return results


#######################
# run_store() function
#######################

def run_store(config, full=False):

   ######################
   # Pydoc documentation
   ######################

   """
   Executes program "store" functionality.

   Also accepts the 'full' argument, which indicates that a new disc
   should be started for this backup, regardless of other configuration.
   This allows for "do a full backup now" and "start a new disc now"
   functionality.
   """

   ############
   # Variables
   ############

   results             = { }
   results['output']   = None
   results['errors']   = None
   results['warnings'] = None

   output              = [ ]
   errors              = [ ]
   warnings            = [ ]

   iso_file            = ""


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      try:

         ##############################
         # Check the staging directory
         ##############################

         # Build the directory name
         daily_dir = os.path.join(config['store']['source_dir'], time.strftime(DIR_TIME_FORMAT))

         # Check to see that the directory exists
         if not os.path.isdir(daily_dir):
            errors.append("Error: expected source directory '%s' does not exist.\n" % daily_dir)
            raise CedarBackupError()

         # Display a warning if the directory isn't complete
         if not filesystem.local_indicator_exists(os.path.join(daily_dir, STAGE_INDICATOR)):
            errors.append("Error: Staging directory '%s' was not ready to be stored.\n" % daily_dir)
            raise CedarBackupError()

      
         #####################################
         # Decide whether to start a new disc
         #####################################

         if _today_is_start(config['options']['starting_day']) or full:
            new_disc = True
         else:
            new_disc = False


         ########################
         # Initialize the device
         ########################

         call = cdr.initialize(device=config['store']['target_device'],
                               scsi_id=config['store']['target_scsi_id'],
                               drive_speed=config['store']['drive_speed'],
                               media_type=config['store']['media_type'])

         if call['warnings'] is not None:
            warnings += call['warnings']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()

         if call['device'] is None:
            errors.append("Error: device object was not initialized properly.\n")
            raise CedarBackupError()

         device_dict = call['device']


         ####################################
         # Check capacity vs. required space
         ####################################

         call = cdr.capacity(device_dict=device_dict, new=new_disc)
         if call is None or call['available'] is None:
            errors.append("Error: unable to get device capacity.\n")
            raise CedarBackupError()

         available = call['available']
         boundaries = call['boundaries']

         required = filesystem.tree_size(daily_dir)
         if required is None:
            errors.append("Error: unable to get directory size.\n")
            raise CedarBackupError()

         if required['bytes'] > available['bytes']:
            errors.append("Error: media does not contain sufficient capacity to store directory.\n")
            errors.append("       Required %f kB (%f MB), available %f kB (%f MB).\n" % (required['kbytes'], required['mbytes'],
                                                                                         available['kbytes'], available['mbytes'] ))
            raise CedarBackupError()


         #######################
         # Create the ISO image
         #######################

         tempfile.tempdir = config['options']['working_dir']
         iso_file = tempfile.mktemp()

         prefix = time.strftime(PREFIX_TIME_FORMAT)
         dir_list = [ { 'prefix': prefix, 'source_dir': daily_dir } ]

         call = cdr.create_image(iso_image=iso_file, 
                                 device_dict=device_dict, 
                                 dir_list=dir_list,
                                 boundaries=boundaries)

         if call['output'] is not None:
            output += call['output']

         if call['warnings'] is not None:
            warnings += call['warnings']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()


         ######################
         # Write the ISO image
         ######################

         # Note that we always try to write multisession, and the underlying
         # CDR code is smart enough to fall back on single-session if the drive
         # doesn't support multisession.
         # 
         # Another subtle point: if we don't have any boundaries (which will
         # happen if the disc in the drive is not already multi-session), then
         # we effectively have to start a new disc.  The write is not going
         # to succeed if we start at the beginning but don't blank the disc
         # first (at least, in the case of CD-RW media).

         if boundaries is None:
            new_disc = True

         call = cdr.write_image(iso_image=iso_file,
                                device_dict=device_dict,
                                new=new_disc,
                                multi=True)
       
         if call['output'] is not None:
            output += call['output']

         if call['warnings'] is not None:
            warnings += call['warnings']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()


         ##################################################
         # Run a consistency check, if configured to do so
         ##################################################

         if config['store']['check_data']:

            call = filesystem.consistency_check(config['options']['working_dir'], 
                                                config['store']['target_device'], 
                                                config['store']['source_dir'], 
                                                [time.strftime(DIR_TIME_FORMAT)])

            if call['output'] is not None:
               output += call['output']
      
            if call['warnings'] is not None:
               warnings += call['warnings']
      
            if call['errors'] is not None:
               errors += call['errors']
               raise CedarBackupError()


      #####################################
      # Handle all Cedar Backup exceptions
      #####################################

      except CedarBackupError:
         pass


   ###########
   # Clean up
   ###########
   # This is done in a finally statement so we're sure it happens.

   finally:
   
      if os.path.isfile(iso_file):
         try:
            os.remove(iso_file)
         except:
            errors.append("Error: unable to remove temporary ISO file '%s'.\n" % iso_file)


   #################
   # Return results
   #################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors;
   if len(warnings) > 0:
      results['warnings'] = warnings
      
   return results


#######################
# run_purge() function
#######################

def run_purge(config):

   ######################
   # Pydoc documentation
   ######################

   """
   Executes program "purge" functionality.
   """

   ############
   # Variables
   ############

   results             = { }
   results['output']   = None
   results['errors']   = None
   results['warnings'] = None

   output              = [ ]
   errors              = [ ]
   warnings            = [ ]


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      ##############################
      # Purge files and directories
      ##############################

      file_count = 0
      dir_count = 0
      for purge_dir in config['purge']['dir']:
         for f in filesystem.aged_files(purge_dir['abs_path'], purge_dir['retain_days']):
            try:
               os.remove(f)
               file_count += 1
               output.append("Removed old file %s.\n" % f)
            except:
               errors.append("Error: unable to remove old file '%s'.\n" % f)
         for d in filesystem.empty_dirs(purge_dir['abs_path']):
            try:
               if d != purge_dir['abs_path']:
                  os.rmdir(d)
                  dir_count += 1
                  output.append("Removed empty directory %s.\n" % d)
            except:
               errors.append("Error: unable to remove empty directory '%s'.\n" % d)

      output.append("Removed %d old files and %d empty directories.\n" % (file_count, dir_count))


   #####################################
   # Handle all Cedar Backup exceptions
   #####################################

   except CedarBackupError:
      pass


   #################
   # Return results
   #################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors;
   if len(warnings) > 0:
      results['warnings'] = warnings
      
   return results


#########################
# run_rebuild() function
#########################

def run_rebuild(config):

   ######################
   # Pydoc documentation
   ######################

   """
   Rebuilds "this week's" disc, if possible, based on staging directory.

   This function exists mainly to recreate a disc that has been "trashed" 
   due to media or hardware problems.  Note that the "stage complete" 
   indicator isn't checked for this function.
   """

   ############
   # Variables
   ############

   results             = { }
   results['output']   = None
   results['errors']   = None
   results['warnings'] = None

   output              = [ ]
   errors              = [ ]
   warnings            = [ ]
   
   iso_file            = ""


   #####################################################
   # Use a big try statement, for easier error-handling
   #####################################################

   try:

      try:

         ##############################
         # Check the staging directory
         ##############################

         # Check to see that the directory exists
         if not os.path.isdir(config['store']['source_dir']):
            errors.append("Error: expected source directory '%s' does not exist.\n" % config['store']['source_dir'])
            raise CedarBackupError()


         ########################
         # Initialize the device
         ########################

         call = cdr.initialize(device=config['store']['target_device'],
                               scsi_id=config['store']['target_scsi_id'],
                               drive_speed=config['store']['drive_speed'],
                               media_type=config['store']['media_type'])

         if call['warnings'] is not None:
            warnings += call['warnings']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()

         if call['device'] is None:
            errors.append("Error: device object was not initialized properly.\n")
            raise CedarBackupError()

         device_dict = call['device']


         ########################################
         # Create the list of backup directories
         ########################################

         # Each date has its own directory, and should be prefixed accordingly.
         # What we're doing here is exactly the same as for the store function,
         # except there the prefix/directory is just "today", whatever that is,
         # and here we have a list of dates.

         dir_list = [ ]
         for date in _this_week_dates(config['options']['starting_day']):
            daily_dir = os.path.join(config['store']['source_dir'], date)
            if os.path.isdir(daily_dir):
               dir_list.append({ 'prefix':date, 'source_dir':daily_dir } )

         if len(dir_list) == 0:
            errors.append("Error: rebuild requested but no staged directories exist for current week.\n")
            raise CedarBackupError()

      
         ####################################
         # Check capacity vs. required space
         ####################################

         call = cdr.capacity(device_dict=device_dict, new=True)
         if call is None or call['available'] is None:
            errors.append("Error: unable to get device capacity.\n")
            raise CedarBackupError()

         available = call['available']
         boundaries = call['boundaries']

         overall           = { }
         overall['bytes']  = 0
         overall['mbytes'] = 0

         for d in dir_list:
            required = filesystem.tree_size(d['source_dir'])
            if required is None:
               errors.append("Error: unable to get directory size for '%s'.\n" % d['source_dir'])
               raise CedarBackupError()
            overall['bytes'] += required['bytes']
            overall['mbytes'] += required['mbytes']

         if overall['bytes'] > available['bytes']:
            errors.append("Error: media does not contain sufficient capacity to store directory.\n")
            errors.append("       Required %f kB (%f MB), available %f kB (%f MB).\n" % (overall['kbytes'], overall['mbytes'],
                                                                                         available['kbytes'], available['mbytes'] ))
            raise CedarBackupError()


         #######################
         # Create the ISO image
         #######################

         tempfile.tempdir = config['options']['working_dir']
         iso_file = tempfile.mktemp()

         call = cdr.create_image(iso_image=iso_file, 
                                 device_dict=device_dict, 
                                 dir_list=dir_list,
                                 boundaries=boundaries)

         if call['output'] is not None:
            output += call['output']

         if call['warnings'] is not None:
            warnings += call['warnings']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()


         ######################
         # Write the ISO image
         ######################
         # Note that we always try to write multisession, and the underlying
         # CDR code is smart enough to fall back on single-session if the drive
         # doesn't support multisession.

         call = cdr.write_image(iso_image=iso_file,
                                device_dict=device_dict,
                                new=True,
                                multi=True)
       
         if call['output'] is not None:
            output += call['output']

         if call['warnings'] is not None:
            warnings += call['warnings']

         if call['errors'] is not None:
            errors += call['errors']
            raise CedarBackupError()


         ##################################################
         # Run a consistency check, if configured to do so
         ##################################################

         if config['store']['check_data']:

            call = filesystem.consistency_check(config['options']['working_dir'], 
                                                config['store']['target_device'], 
                                                config['store']['source_dir'], 
                                                _this_week_dates(config['options']['starting_day']))


            if call['output'] is not None:
               output += call['output']
      
            if call['warnings'] is not None:
               warnings += call['warnings']
      
            if call['errors'] is not None:
               errors += call['errors']
               raise CedarBackupError()


      #####################################
      # Handle all Cedar Backup exceptions
      #####################################

      except CedarBackupError:
         pass


   ###########
   # Clean up
   ###########
   # This is done in a finally statement so we're sure it happens.

   finally:

      if os.path.isfile(iso_file):
         try:
            os.remove(iso_file)
         except:
            errors.append("Error: unable to remove temporary ISO file '%s'.\n" % iso_file)


   #################
   # Return results
   #################

   if len(output) > 0:
      results['output'] = output
   if len(errors) > 0:
      results['errors'] = errors;
   if len(warnings) > 0:
      results['warnings'] = warnings
      
   return results


########################################################################
# Private functions
########################################################################

##################################
# _convert_day_of_week() function
##################################

def _convert_day_of_week(day_name):
   """Converts a day-of-week string (i.e. "monday") to a number (in this case, 0)."""
   if string.lower(day_name) == "monday":
      return 0
   elif string.lower(day_name) == "tuesday":
      return 1
   elif string.lower(day_name) == "wednesday":
      return 2
   elif string.lower(day_name) == "thursday":
      return 3
   elif string.lower(day_name) == "friday":
      return 4
   elif string.lower(day_name) == "saturday":
      return 5
   elif string.lower(day_name) == "sunday":
      return 6
   else:
      return -1  # What else can we do??


#############################
# _today_is_start() function
#############################

def _today_is_start(start_of_week):
   """Checks whether today is the starting day.  Returns True or False."""
   return time.localtime()[6] == _convert_day_of_week(start_of_week)


##############################
# _this_week_dates() function
##############################

def _this_week_dates(start_of_week):

   """
   Answers the question "If we start backups on Tuesday, and today is
   Thursday, what dates YYYY/MM/DD are on this week's disc?"
   """

   # We use mktime()'s ability to normalize dates, so that if we take 
   # 2002/01/01 and subtract a day, we get 2001/12/31.

   start_dow = _convert_day_of_week(start_of_week)
   today_dow = time.localtime()[6]

   if today_dow >= start_dow:
      days = (today_dow - start_dow) + 1
   else:
      days = 7 - (start_dow - today_dow) + 1

   dates = [ ]
   for i in range(0, days):
      struct_tm = map(None, time.localtime())
      struct_tm[2] -= i
      dates.append(time.strftime(PREFIX_TIME_FORMAT, time.localtime(time.mktime(struct_tm))))

   return dates


########################################################################
# Module entry point
########################################################################

# Ensures that the module isn't run if someone just imports it.
if __name__ == '__main__':
   print "Module just exists to be imported, sorry."


