#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-
# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
#              C E D A R
#          S O L U T I O N S       "Software done right."
#           S O F T W A R E
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Copyright (c) 2004-2005 Kenneth J. Pronovici.
# All rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# Version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Copies of the GNU General Public License are available from
# the Free Software Foundation website, http://www.gnu.org/.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.3)
# Project  : Cedar Backup, release 2
# Revision : $Id: filesystemtests.py 696 2005-03-10 23:21:27Z pronovic $
# Purpose  : Tests filesystem-related classes.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# This file was created with a width of 132 characters, and NO tabs.

########################################################################
# Module documentation
########################################################################

"""
Unit tests for CedarBackup2/filesystem.py.

Test Notes
==========

   This module contains individual tests for each of the classes implemented in
   filesystem.py: FilesystemList, BackupFileList and PurgeItemList.

   The BackupFileList and PurgeItemList classes inherit from FilesystemList,
   and the FilesystemList class itself inherits from the standard Python list
   class.  For the most part, I won't spend time testing inherited
   functionality, especially if it's already been tested.  However, I do test
   some of the base list functionality just to ensure that the inheritence has
   been constructed properly and everything seems to work as expected.

   You may look at this code and ask, "Why all of the checks that XXX is in
   list YYY?  Why not just compare what we got to a known list?"  The answer is
   that the order of the list is not significant, only its contents.  We can't
   be positive about the order in which we recurse a directory, but we do need
   to make sure that everything we expect is in the list and nothing more.  We
   do this by checking the count if items and then making sure that exactly
   that many known items exist in the list.  

   This file is ridiculously long, almost too long to be worked with easily.  I
   really should split it up into smaller files, but I like having a 1:1
   relationship between a module and its test.

Naming Conventions
==================

   I prefer to avoid large unit tests which validate more than one piece of
   functionality.  Instead, I create lots of very small tests that each
   validate one specific thing.  These small tests are then named with an index
   number, yielding something like C{testAddDir_001} or C{testValidate_023}.
   Each method then has a docstring describing what it's supposed to
   accomplish.  I feel that this makes it easier to judge the extent of a
   problem when one exists.

Full vs. Reduced Tests
======================

   All of the tests in this module are considered safe to be run in an average
   build environment.  There is a no need to use a FILESYSTEMTESTS_FULL
   environment variable to provide a "reduced feature set" test suite as for
   some of the other test modules.

@author Kenneth J. Pronovici <pronovic@ieee.org>
"""


########################################################################
# Import modules and do runtime validations
########################################################################

import sys
import os
import sha
import time
import unittest
import tempfile
import tarfile

from CedarBackup2.testutil import findResources, buildPath, removedir, extractTar, changeFileAge, randomFilename
from CedarBackup2.filesystem import FilesystemList, BackupFileList, PurgeItemList, normalizeDir, compareContents


#######################################################################
# Module-wide configuration and constants
#######################################################################

DATA_DIRS = [ "./data", "./test/data" ]
RESOURCES = [ "tree1.tar.gz", "tree2.tar.gz", "tree3.tar.gz", "tree4.tar.gz", "tree5.tar.gz",
              "tree6.tar.gz", "tree7.tar.gz", "tree8.tar.gz", "tree9.tar.gz", "tree10.tar.gz", 
              "tree11.tar.gz", "tree12.tar.gz", "tree13.tar.gz", ]

INVALID_FILE      = "bogus"         # This file name should never exist
NOMATCH_PATH      = "/something"    # This file should never match something we put in a file list 
NOMATCH_PATTERN   = "pattern"       # This pattern should never match something we put in a file list 

AGE_1_HOUR        = 1*60*60         # in seconds
AGE_2_HOURS       = 2*60*60         # in seconds
AGE_12_HOURS      = 12*60*60        # in seconds
AGE_23_HOURS      = 23*60*60        # in seconds
AGE_24_HOURS      = 24*60*60        # in seconds
AGE_25_HOURS      = 25*60*60        # in seconds
AGE_47_HOURS      = 47*60*60        # in seconds
AGE_48_HOURS      = 48*60*60        # in seconds
AGE_49_HOURS      = 49*60*60        # in seconds


#######################################################################
# Test Case Classes
#######################################################################

###########################
# TestFilesystemList class
###########################

class TestFilesystemList(unittest.TestCase):

   """Tests for the FilesystemList class."""

   ################
   # Setup methods
   ################

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.resources = findResources(RESOURCES, DATA_DIRS)
      except Exception, e:
         self.fail(e)

   def tearDown(self):
      removedir(self.tmpdir)


   ##################
   # Utility methods
   ##################

   def extractTar(self, tarname):
      """Extracts a tarfile with a particular name."""
      extractTar(self.tmpdir, self.resources['%s.tar.gz' % tarname])

   def buildPath(self, components):
      """Builds a complete search path from a list of components."""
      components.insert(0, self.tmpdir)
      return buildPath(components)


   ################################
   # Test basic list functionality
   ################################
         
   def testBasic_001(self):
      """
      Test the append() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('a')
      self.failUnlessEqual(['a'], fsList)
      fsList.append('b')
      self.failUnlessEqual(['a', 'b'], fsList)

   def testBasic_002(self):
      """
      Test the insert() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.insert(0, 'a')
      self.failUnlessEqual(['a'], fsList)
      fsList.insert(0, 'b')
      self.failUnlessEqual(['b', 'a'], fsList)

   def testBasic_003(self):
      """
      Test the remove() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.insert(0, 'a')
      fsList.insert(0, 'b')
      self.failUnlessEqual(['b', 'a'], fsList)
      fsList.remove('a')
      self.failUnlessEqual(['b'], fsList)
      fsList.remove('b')
      self.failUnlessEqual([], fsList)

   def testBasic_004(self):
      """
      Test the pop() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('a')
      fsList.append('b')
      fsList.append('c')
      fsList.append('d')
      fsList.append('e')
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)
      self.failUnlessEqual('e', fsList.pop())
      self.failUnlessEqual(['a', 'b', 'c', 'd'], fsList)
      self.failUnlessEqual('d', fsList.pop())
      self.failUnlessEqual(['a', 'b', 'c'], fsList)
      self.failUnlessEqual('c', fsList.pop())
      self.failUnlessEqual(['a', 'b'], fsList)
      self.failUnlessEqual('b', fsList.pop())
      self.failUnlessEqual(['a'], fsList)
      self.failUnlessEqual('a', fsList.pop())
      self.failUnlessEqual([], fsList)

   def testBasic_005(self):
      """
      Test the count() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('a')
      fsList.append('b')
      fsList.append('c')
      fsList.append('d')
      fsList.append('e')
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)
      self.failUnlessEqual(1, fsList.count('a'))

   def testBasic_006(self):
      """
      Test the index() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('a')
      fsList.append('b')
      fsList.append('c')
      fsList.append('d')
      fsList.append('e')
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)
      self.failUnlessEqual(2, fsList.index('c'))

   def testBasic_007(self):
      """
      Test the reverse() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('a')
      fsList.append('b')
      fsList.append('c')
      fsList.append('d')
      fsList.append('e')
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)
      fsList.reverse()
      self.failUnlessEqual(['e', 'd', 'c', 'b', 'a'], fsList)
      fsList.reverse()
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)

   def testBasic_008(self):
      """
      Test the sort() method.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('e')
      fsList.append('d')
      fsList.append('c')
      fsList.append('b')
      fsList.append('a')
      self.failUnlessEqual(['e', 'd', 'c', 'b', 'a'], fsList)
      fsList.sort()
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)
      fsList.sort()
      self.failUnlessEqual(['a', 'b', 'c', 'd', 'e'], fsList)

   def testBasic_009(self):
      """
      Test slicing.
      """
      fsList = FilesystemList()
      self.failUnlessEqual([], fsList)
      fsList.append('e')
      fsList.append('d')
      fsList.append('c')
      fsList.append('b')
      fsList.append('a')
      self.failUnlessEqual(['e', 'd', 'c', 'b', 'a'], fsList)
      self.failUnlessEqual(['e', 'd', 'c', 'b', 'a'], fsList[:])
      self.failUnlessEqual(['e', 'd', 'c', 'b', 'a'], fsList[0:])
      self.failUnlessEqual('e', fsList[0])
      self.failUnlessEqual('a', fsList[4])
      self.failUnlessEqual(['d', 'c', 'b'], fsList[1:4])


   #################
   # Test addFile()
   #################
         
   def testAddFile_001(self):
      """
      Attempt to add a file that doesn't exist; no exclusions.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_002(self):
      """
      Attempt to add a directory; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_003(self):
      """
      Attempt to add a soft link; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_004(self):
      """
      Attempt to add an existing file; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddFile_005(self):
      """
      Attempt to add a file that doesn't exist; excludeFiles set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_006(self):
      """
      Attempt to add a directory; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_007(self):
      """
      Attempt to add a soft link; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_008(self):
      """
      Attempt to add an existing file; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddFile_009(self):
      """
      Attempt to add a file that doesn't exist; excludeDirs set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_010(self):
      """
      Attempt to add a directory; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_011(self):
      """
      Attempt to add a soft link; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_012(self):
      """
      Attempt to add an existing file; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddFile_013(self):
      """
      Attempt to add a file that doesn't exist; excludeLinks set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_014(self):
      """
      Attempt to add a directory; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_015(self):
      """
      Attempt to add a soft link; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_016(self):
      """
      Attempt to add an existing file; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddFile_017(self):
      """
      Attempt to add a file that doesn't exist; with excludePaths including the
      path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_018(self):
      """
      Attempt to add a directory; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_019(self):
      """
      Attempt to add a soft link; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_020(self):
      """
      Attempt to add an existing file; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddFile_021(self):
      """
      Attempt to add a file that doesn't exist; with excludePaths not including
      the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_022(self):
      """
      Attempt to add a directory; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_023(self):
      """
      Attempt to add a soft link; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_024(self):
      """
      Attempt to add an existing file; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddFile_025(self):
      """
      Attempt to add a file that doesn't exist; with excludePatterns matching
      the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_026(self):
      """
      Attempt to add a directory; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_027(self):
      """
      Attempt to add a soft link; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_028(self):
      """
      Attempt to add an existing file; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addFile(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddFile_029(self):
      """
      Attempt to add a file that doesn't exist; with excludePatterns not
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ] 
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_030(self):
      """
      Attempt to add a directory; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_031(self):
      """
      Attempt to add a soft link; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_032(self):
      """
      Attempt to add an existing file; with excludePatterns not matching the
      path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddFile_033(self):
      """
      Attempt to add an invalid link (i.e. a link that points to something that
      doesn't exist).
      """
      self.extractTar("tree10")
      path = self.buildPath(["tree10", "link001"])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addFile, path)

   def testAddFile_034(self):
      """
      Attempt to add a file that has spaces in its name.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", "file with spaces"])
      fsList = FilesystemList()
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddFile_035(self):
      """
      Attempt to add a UTF-8 file.
      """
      self.extractTar("tree12")
      path = self.buildPath(["tree12", "unicode", "\xe2\x99\xaa\xe2\x99\xac"])
      fsList = FilesystemList()
      count = fsList.addFile(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)


   ################
   # Test addDir()
   ################
         
   def testAddDir_001(self):
      """
      Attempt to add a directory that doesn't exist; no exclusions.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_002(self):
      """
      Attempt to add a file; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_003(self):
      """
      Attempt to add a soft link; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_004(self):
      """
      Attempt to add an existing directory; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_005(self):
      """
      Attempt to add a directory that doesn't exist; excludeFiles set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_006(self):
      """
      Attempt to add a file; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_007(self):
      """
      Attempt to add a soft link; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_008(self):
      """
      Attempt to add an existing directory; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_009(self):
      """
      Attempt to add a directory that doesn't exist; excludeDirs set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_010(self):
      """
      Attempt to add a file; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_011(self):
      """
      Attempt to add a soft link; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_012(self):
      """
      Attempt to add an existing directory; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_013(self):
      """
      Attempt to add a directory that doesn't exist; excludeLinks set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_014(self):
      """
      Attempt to add a file; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_015(self):
      """
      Attempt to add a soft link; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_016(self):
      """
      Attempt to add an existing directory; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_017(self):
      """
      Attempt to add a directory that doesn't exist; with excludePaths
      including the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_018(self):
      """
      Attempt to add a file; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_019(self):
      """
      Attempt to add a soft link; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_020(self):
      """
      Attempt to add an existing directory; with excludePaths including the
      path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_021(self):
      """
      Attempt to add a directory that doesn't exist; with excludePaths not
      including the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_022(self):
      """
      Attempt to add a file; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_023(self):
      """
      Attempt to add a soft link; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_024(self):
      """
      Attempt to add an existing directory; with excludePaths not including the
      path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_025(self):
      """
      Attempt to add a directory that doesn't exist; with excludePatterns
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_026(self):
      """
      Attempt to add a file; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_027(self):
      """
      Attempt to add a soft link; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_028(self):
      """
      Attempt to add an existing directory; with excludePatterns matching the
      path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDir(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDir_029(self):
      """
      Attempt to add a directory that doesn't exist; with excludePatterns not
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_030(self):
      """
      Attempt to add a file; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_031(self):
      """
      Attempt to add a soft link; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDir, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_032(self):
      """
      Attempt to add an existing directory; with excludePatterns not matching
      the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDir_033(self):
      """
      Attempt to add an invalid link (i.e. a link that points to something that
      doesn't exist).
      """
      self.extractTar("tree10")
      path = self.buildPath(["tree10", "link001"])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDir, path)

   def testAddDir_034(self):
      """
      Attempt to add a directory that has spaces in its name.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", "dir with spaces"])
      fsList = FilesystemList()
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)


   ########################
   # Test addDirContents()
   ########################
         
   def testAddDirContents_001(self):
      """
      Attempt to add a directory that doesn't exist; no exclusions.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_002(self):
      """
      Attempt to add a file; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_003(self):
      """
      Attempt to add a soft link; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      count = fsList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_004(self):
      """
      Attempt to add an empty directory containing ignore file; no exclusions.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_005(self):
      """
      Attempt to add an empty directory; no exclusions.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_006(self):
      """
      Attempt to add an non-empty directory containing ignore file; no
      exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_007(self):
      """
      Attempt to add an non-empty directory; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath(["tree5", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in fsList)

   def testAddDirContents_008(self):
      """
      Attempt to add a directory that doesn't exist; excludeFiles set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_009(self):
      """
      Attempt to add a file; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_010(self):
      """
      Attempt to add a soft link; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeFiles = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_011(self):
      """
      Attempt to add an empty directory containing ignore file; excludeFiles set.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludeFiles = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_012(self):
      """
      Attempt to add an empty directory; excludeFiles set.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_013(self):
      """
      Attempt to add an non-empty directory containing ignore file; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludeFiles = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_014(self):
      """
      Attempt to add an non-empty directory; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(5, count)
      self.failUnlessEqual(5, len(fsList))
      self.failUnless(self.buildPath(["tree5", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in fsList)

   def testAddDirContents_015(self):
      """
      Attempt to add a directory that doesn't exist; excludeDirs set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_016(self):
      """
      Attempt to add a file; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_017(self):
      """
      Attempt to add a soft link; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeDirs = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_018(self):
      """
      Attempt to add an empty directory containing ignore file; excludeDirs set.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludeDirs = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_019(self):
      """
      Attempt to add an empty directory; excludeDirs set.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_020(self):
      """
      Attempt to add an non-empty directory containing ignore file; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludeDirs = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_021(self):
      """
      Attempt to add an non-empty directory; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(3, len(fsList))
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in fsList)

   def testAddDirContents_023(self):
      """
      Attempt to add a directory that doesn't exist; excludeLinks set.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_024(self):
      """
      Attempt to add a file; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_025(self):
      """
      Attempt to add a soft link; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludeLinks = True
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_026(self):
      """
      Attempt to add an empty directory containing ignore file; excludeLinks set.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludeLinks = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_027(self):
      """
      Attempt to add an empty directory; excludeLinks set.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnless(self.buildPath(["tree8", "dir001", ]) in fsList)

   def testAddDirContents_028(self):
      """
      Attempt to add an non-empty directory containing ignore file; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludeLinks = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_029(self):
      """
      Attempt to add an non-empty directory; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(7, len(fsList))
      self.failUnless(self.buildPath(["tree5", "dir001", ]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in fsList)

   def testAddDirContents_030(self):
      """
      Attempt to add a directory that doesn't exist; with excludePaths
      including the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_031(self):
      """
      Attempt to add a file; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_032(self):
      """
      Attempt to add a soft link; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_033(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePaths including the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePaths = [ path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_034(self):
      """
      Attempt to add an empty directory; with excludePaths including the path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_035(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePaths = [ path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_036(self):
      """
      Attempt to add an non-empty directory; with excludePaths including the
      main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_037(self):
      """
      Attempt to add a directory that doesn't exist; with excludePaths not
      including the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_038(self):
      """
      Attempt to add a file; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_039(self):
      """
      Attempt to add a soft link; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_040(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePaths not including the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_041(self):
      """
      Attempt to add an empty directory; with excludePaths not including the
      path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_042(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_043(self):
      """
      Attempt to add an non-empty directory; with excludePaths not including
      the main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePaths = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath(["tree5", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in fsList)

   def testAddDirContents_044(self):
      """
      Attempt to add a directory that doesn't exist; with excludePatterns
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_045(self):
      """
      Attempt to add a file; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_046(self):
      """
      Attempt to add a soft link; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_047(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePatterns matching the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_048(self):
      """
      Attempt to add an empty directory; with excludePatterns matching the
      path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_049(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_050(self):
      """
      Attempt to add an non-empty directory; with excludePatterns matching the
      main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*%s.*" % path ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_051(self):
      """
      Attempt to add a directory that doesn't exist; with excludePatterns not
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_052(self):
      """
      Attempt to add a file; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_053(self):
      """
      Attempt to add a soft link; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_054(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePatterns not matching the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_055(self):
      """
      Attempt to add an empty directory; with excludePatterns not matching the
      path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], fsList)

   def testAddDirContents_056(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], fsList)

   def testAddDirContents_057(self):
      """
      Attempt to add an non-empty directory; with excludePatterns not matching
      the main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ NOMATCH_PATH ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath(["tree5", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in fsList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in fsList)

   def testAddDirContents_058(self):
      """
      Attempt to add a large tree with no exclusions.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(136, count)
      self.failUnlessEqual(136, len(fsList))
      self.failUnless(self.buildPath([ "tree6", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in fsList)

   def testAddDirContents_059(self):
      """
      Attempt to add a large tree, with excludeFiles set.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      fsList.excludeFiles = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(42, count)
      self.failUnlessEqual(42, len(fsList))
      self.failUnless(self.buildPath([ "tree6", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in fsList)

   def testAddDirContents_060(self):
      """
      Attempt to add a large tree, with excludeDirs set.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      fsList.excludeDirs = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(94, count)
      self.failUnlessEqual(94, len(fsList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in fsList)

   def testAddDirContents_061(self):
      """
      Attempt to add a large tree, with excludeLinks set.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      fsList.excludeLinks = True
      count = fsList.addDirContents(path)
      self.failUnlessEqual(96, count)
      self.failUnlessEqual(96, len(fsList))
      self.failUnless(self.buildPath([ "tree6", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in fsList)

   def testAddDirContents_062(self):
      """
      Attempt to add a large tree, with excludePaths set to exclude some entries.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      fsList.excludePaths = [ self.buildPath([ "tree6", "dir001", "dir002", ]), 
                              self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]), 
                              self.buildPath([ "tree6", "dir003", "dir002", "file001", ]), 
                              self.buildPath([ "tree6", "dir003", "dir002", "file002", ]), ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(125, count)
      self.failUnlessEqual(125, len(fsList))
      self.failUnless(self.buildPath([ "tree6", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in fsList)

   def testAddDirContents_063(self):
      """
      Attempt to add a large tree, with excludePatterns set to exclude some entries.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      fsList.excludePatterns = [ ".*file001.*", ".*tree6\/dir002\/dir001.*" ]
      count = fsList.addDirContents(path)
      self.failUnlessEqual(108, count)
      self.failUnlessEqual(108, len(fsList))
      self.failUnless(self.buildPath([ "tree6", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in fsList)

   def testAddDirContents_064(self):
      """
      Attempt to add a large tree, with ignoreFile set to exclude some directories.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      fsList = FilesystemList()
      fsList.ignoreFile = "ignore"
      count = fsList.addDirContents(path)
      self.failUnlessEqual(79, count)
      self.failUnlessEqual(79, len(fsList))
      self.failUnless(self.buildPath([ "tree6", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in fsList)

   def testAddDirContents_065(self):
      """
      Attempt to add a link to a file.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9", "dir002", "link003", ])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_066(self):
      """
      Attempt to add a link to a directory (which should add its contents).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9", "link002" ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(9, count)
      self.failUnlessEqual(9, len(fsList))
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link004", ]) in fsList)

   def testAddDirContents_067(self):
      """
      Attempt to add an invalid link (i.e. a link that points to something that
      doesn't exist).
      """
      self.extractTar("tree10")
      path = self.buildPath(["tree10", "link001"])
      fsList = FilesystemList()
      self.failUnlessRaises(ValueError, fsList.addDirContents, path)

   def testAddDirContents_068(self):
      """
      Attempt to add directory containing an invalid link (i.e. a link that
      points to something that doesn't exist).
      """
      self.extractTar("tree10")
      path = self.buildPath(["tree10"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(3, len(fsList))
      self.failUnless(self.buildPath([ "tree10", ]) in fsList)
      self.failUnless(self.buildPath([ "tree10", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree10", "dir002", ]) in fsList)

   def testAddDirContents_069(self):
      """
      Attempt to add a directory containing items with spaces.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testAddDirContents_070(self):
      """
      Attempt to add a directory which has a name containing spaces.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", "dir with spaces", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(5, count)
      self.failUnlessEqual(5, len(fsList))
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testAddDirContents_071(self):
      """
      Attempt to add a directory which has a UTF-8 filename in it.
      """
      self.extractTar("tree12")
      path = self.buildPath(["tree12", "unicode", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(6, len(fsList))
      self.failUnless(self.buildPath([ "tree12", "unicode", ]) in fsList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "README.strange-name", ]) in fsList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "utflist.long.gz", ]) in fsList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "utflist.cp437.gz", ]) in fsList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "utflist.short.gz", ]) in fsList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "\xe2\x99\xaa\xe2\x99\xac", ]) in fsList)

   def testAddDirContents_072(self):
      """
      Attempt to add a directory which has several UTF-8 filenames in it.
      This test data was taken from Rick Lowe's problems around the release of v1.10.
      """
      self.extractTar("tree13")
      path = self.buildPath(["tree13", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree13", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "Les mouvements de r\x82forme.doc", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "l'\x82nonc\x82.sxw", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard - renvois et bibliographie.sxw", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard copie finale.sxw", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard de vinci - page titre.sxw", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard de vinci.sxw", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "Rammstein - B\x81ck Dich.mp3", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "megaherz - Glas Und Tr\x84nen.mp3", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "Megaherz - Mistst\x81ck.MP3", ]) in fsList)
      self.failUnless(self.buildPath([ "tree13", "Rammstein - Mutter - B\x94se.mp3", ]) in fsList)


   #####################
   # Test removeFiles()
   #####################
         
   def testRemoveFiles_001(self):
      """
      Test with an empty list and a pattern of None.
      """
      fsList = FilesystemList()
      count = fsList.removeFiles(pattern=None)
      self.failUnlessEqual(0, count)

   def testRemoveFiles_002(self):
      """
      Test with an empty list and a non-empty pattern.
      """
      fsList = FilesystemList()
      count = fsList.removeFiles(pattern="pattern")
      self.failUnlessEqual(0, count)

   def testRemoveFiles_003(self):
      """
      Test with a non-empty list (files only) and a pattern of None.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=None)
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(1, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)

   def testRemoveFiles_004(self):
      """
      Test with a non-empty list (directories only) and a pattern of None.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeFiles(pattern=None)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveFiles_005(self):
      """
      Test with a non-empty list (files and directories) and a pattern of None.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=None)
      self.failUnlessEqual(44, count)
      self.failUnlessEqual(37, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)

   def testRemoveFiles_006(self):
      """
      Test with a non-empty list (files, directories and links) and
      a pattern of None.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeFiles(pattern=None)
      self.failUnlessEqual(10, count)
      self.failUnlessEqual(12, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveFiles_007(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a pattern of None.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=None)
      self.failUnlessEqual(44, count)
      self.failUnlessEqual(38, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)

   def testRemoveFiles_008(self):
      """
      Test with a non-empty list (spaces in path names) and a pattern of None.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveFiles_009(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches none of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveFiles_010(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveFiles_011(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveFiles_012(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveFiles_013(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveFiles_014(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeFiles(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveFiles_015(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches some of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=".*tree1.*file00[67]")
      self.failUnlessEqual(2, count)
      self.failUnlessEqual(6, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)

   def testRemoveFiles_016(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeFiles(pattern=".*tree2.*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveFiles_017(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=".*tree4.*dir006.*")
      self.failUnlessEqual(10, count)
      self.failUnlessEqual(71, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveFiles_018(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeFiles(pattern=".*tree9.*dir002.*")
      self.failUnlessEqual(4, count)
      self.failUnlessEqual(18, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveFiles_019(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=".*dir001.*file002.*")
      self.failUnlessEqual(1, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveFiles_020(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeFiles(pattern=".*with spaces.*")
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(10, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)

   def testRemoveFiles_021(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches all of the entries.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=".*")
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(1, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)

   def testRemoveFiles_022(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches all of the entries.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeFiles(pattern=".*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveFiles_023(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=".*")
      self.failUnlessEqual(44, count)
      self.failUnlessEqual(37, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)

   def testRemoveFiles_024(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeFiles(pattern=".*")
      self.failUnlessEqual(10, count)
      self.failUnlessEqual(12, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveFiles_025(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeFiles(pattern=".*")
      self.failUnlessEqual(44, count)
      self.failUnlessEqual(38, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)

   def testRemoveFiles_026(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeFiles(pattern=".*")
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(5, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)


   ####################
   # Test removeDirs()
   ####################
         
   def testRemoveDirs_001(self):
      """
      Test with an empty list and a pattern of None.
      """
      fsList = FilesystemList()
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(0, count)

   def testRemoveDirs_002(self):
      """
      Test with an empty list and a non-empty pattern.
      """
      fsList = FilesystemList()
      count = fsList.removeDirs(pattern="pattern")
      self.failUnlessEqual(0, count)

   def testRemoveDirs_003(self):
      """
      Test with a non-empty list (files only) and a pattern of None.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual(7, len(fsList))
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveDirs_004(self):
      """
      Test with a non-empty list (directories only) and a pattern of None.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveDirs_005(self):
      """
      Test with a non-empty list (files and directories) and a pattern of None.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(37, count)
      self.failUnlessEqual(44, len(fsList))
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_006(self):
      """
      Test with a non-empty list (files, directories and links) and a pattern of None.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(12, count)
      self.failUnlessEqual(10, len(fsList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)

   def testRemoveDirs_007(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a pattern of None.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(37, count)
      self.failUnlessEqual(45, len(fsList))
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_008(self):
      """
      Test with a non-empty list (spaces in path names) and a pattern of None.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeDirs(pattern=None)
      self.failUnlessEqual(5, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveDirs_009(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches none of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveDirs_010(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveDirs_011(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_012(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveDirs_013(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_014(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeDirs(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveDirs_015(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches some of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=".*tree1.file00[67]")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveDirs_016(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeDirs(pattern=".*dir0[012]0")
      self.failUnlessEqual(1, count)
      self.failUnlessEqual(10, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)

   def testRemoveDirs_017(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=".*dir001")
      self.failUnlessEqual(9, count)
      self.failUnlessEqual(72, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_018(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeDirs(pattern=".*tree9.*dir002.*")
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveDirs_019(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=".*dir001")
      self.failUnlessEqual(9, count)
      self.failUnlessEqual(73, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_020(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeDirs(pattern=".*with spaces.*")
      self.failUnlessEqual(1, count)
      self.failUnlessEqual(15, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveDirs_021(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches all of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=".*")
      self.failUnlessEqual(1, count)
      self.failUnlessEqual(7, len(fsList))
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveDirs_022(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeDirs(pattern=".*")
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveDirs_023(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=".*")
      self.failUnlessEqual(37, count)
      self.failUnlessEqual(44, len(fsList))
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_024(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeDirs(pattern=".*")
      self.failUnlessEqual(12, count)
      self.failUnlessEqual(10, len(fsList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)

   def testRemoveDirs_025(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeDirs(pattern=".*")
      self.failUnlessEqual(37, count)
      self.failUnlessEqual(45, len(fsList))
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveDirs_026(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeDirs(pattern=".*")
      self.failUnlessEqual(5, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)


   #####################
   # Test removeLinks()
   #####################
         
   def testRemoveLinks_001(self):
      """
      Test with an empty list and a pattern of None.
      """
      fsList = FilesystemList()
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(0, count)

   def testRemoveLinks_002(self):
      """
      Test with an empty list and a non-empty pattern.
      """
      fsList = FilesystemList()
      count = fsList.removeLinks(pattern="pattern")
      self.failUnlessEqual(0, count)

   def testRemoveLinks_003(self):
      """
      Test with a non-empty list (files only) and a pattern of None.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveLinks_004(self):
      """
      Test with a non-empty list (directories only) and a pattern of None.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveLinks_005(self):
      """
      Test with a non-empty list (files and directories) and a pattern of None.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_006(self):
      """
      Test with a non-empty list (files, directories and links) and a pattern of None.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(9, count)
      self.failUnlessEqual(13, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)

   def testRemoveLinks_007(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a pattern of None.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_008(self):
      """
      Test with a non-empty list (spaces in path names) and a pattern of None.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeLinks(pattern=None)
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(10, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)

   def testRemoveLinks_009(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches none of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveLinks_010(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveLinks_011(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_012(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveLinks_013(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_014(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeLinks(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveLinks_015(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches some of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=".*tree1.*file007")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveLinks_016(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeLinks(pattern=".*tree2.*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveLinks_017(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=".*tree4.*dir006.*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_018(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeLinks(pattern=".*tree9.*dir002.*")
      self.failUnlessEqual(4, count)
      self.failUnlessEqual(18, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveLinks_019(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=".*tree4.*dir006.*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_020(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeLinks(pattern=".*with spaces.*")
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(13, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)

   def testRemoveLinks_021(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches all of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=".*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveLinks_022(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeLinks(pattern=".*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveLinks_023(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=".*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_024(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeLinks(pattern=".*");
      self.failUnlessEqual(9, count)
      self.failUnlessEqual(13, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)

   def testRemoveLinks_025(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeLinks(pattern=".*")
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveLinks_026(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeLinks(pattern=".*")
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(10, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)


   #####################
   # Test removeMatch()
   #####################
         
   def testRemoveMatch_001(self):
      """
      Test with an empty list and a pattern of None.
      """
      fsList = FilesystemList()
      self.failUnlessRaises(TypeError, fsList.removeMatch, pattern=None)

   def testRemoveMatch_002(self):
      """
      Test with an empty list and a non-empty pattern.
      """
      fsList = FilesystemList()
      count = fsList.removeMatch(pattern="pattern")
      self.failUnlessEqual(0, count)

   def testRemoveMatch_003(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches none of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveMatch_004(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveMatch_005(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveMatch_006(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches none of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveMatch_007(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches none of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveMatch_008(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches none of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeMatch(pattern=NOMATCH_PATTERN)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)

   def testRemoveMatch_009(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches some of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=".*file00[135].*")
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(5, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveMatch_010(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeMatch(pattern=".*dir00[2468].*")
      self.failUnlessEqual(4, count)
      self.failUnlessEqual(7, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveMatch_011(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=".*tree4.*dir006")
      self.failUnlessEqual(18, count)
      self.failUnlessEqual(63, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveMatch_012(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches some of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeMatch(pattern=".*file001.*")
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(19, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveMatch_013(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches some of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=".*dir00[46].*")
      self.failUnlessEqual(25, count)
      self.failUnlessEqual(57, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveMatch_014(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches some of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeMatch(pattern=".*with spaces.*")
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(9, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)

   def testRemoveMatch_015(self):
      """
      Test with a non-empty list (files only) and a non-empty pattern that
      matches all of them.
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=".*")
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveMatch_016(self):
      """
      Test with a non-empty list (directories only) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeMatch(pattern=".*")
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveMatch_017(self):
      """
      Test with a non-empty list (files and directories) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=".*")
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveMatch_019(self):
      """
      Test with a non-empty list (files, directories and links) and a non-empty
      pattern that matches all of them.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeMatch(pattern=".*")
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveMatch_020(self):
      """
      Test with a non-empty list (files and directories, some nonexistent) and
      a non-empty pattern that matches all of them.
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      fsList.append(self.buildPath([ "tree4", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(82, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeMatch(pattern=".*")
      self.failUnlessEqual(82, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveMatch_021(self):
      """
      Test with a non-empty list (spaces in path names) and a non-empty pattern
      that matches all of them.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeMatch(pattern=".*")
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(0, len(fsList))


   #######################
   # Test removeInvalid()
   #######################
         
   def testRemoveInvalid_001(self):
      """
      Test with an empty list.
      """
      fsList = FilesystemList()
      count = fsList.removeInvalid()
      self.failUnlessEqual(0, count)

   def testRemoveInvalid_002(self):
      """
      Test with a non-empty list containing only invalid entries (some with spaces).
      """
      self.extractTar("tree9")
      fsList = FilesystemList()
      fsList.append(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", " %s 5  " % INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(5, len(fsList))
      self.failUnless(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", " %s 5  " % INVALID_FILE, ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(5, count)
      self.failUnlessEqual(0, len(fsList))

   def testRemoveInvalid_003(self):
      """
      Test with a non-empty list containing only valid entries (files only).
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testRemoveInvalid_004(self):
      """
      Test with a non-empty list containing only valid entries (directories
      only).
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testRemoveInvalid_005(self):
      """
      Test with a non-empty list containing only valid entries (files and
      directories).
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testRemoveInvalid_006(self):
      """
      Test with a non-empty list containing only valid entries (files,
      directories and links).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveInvalid_007(self):
      """
      Test with a non-empty list containing valid and invalid entries (files,
      directories and links).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      fsList.append(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(26, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(4, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testRemoveInvalid_008(self):
      """
      Test with a non-empty list containing only valid entries (files,
      directories and links, some with spaces).
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      count = fsList.removeInvalid()
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)


   ###################
   # Test normalize()
   ###################
         
   def testNormalize_001(self):
      """
      Test with an empty list.
      """
      fsList = FilesystemList()
      self.failUnlessEqual(0, len(fsList))
      fsList.normalize()
      self.failUnlessEqual(0, len(fsList))

   def testNormalize_002(self):
      """
      Test with a list containing one entry.
      """
      fsList = FilesystemList()
      fsList.append("one");
      self.failUnlessEqual(1, len(fsList))
      fsList.normalize()
      self.failUnlessEqual(1, len(fsList))
      self.failUnless("one" in fsList)

   def testNormalize_003(self):
      """
      Test with a list containing two entries, no duplicates.
      """
      fsList = FilesystemList()
      fsList.append("one");
      fsList.append("two");
      self.failUnlessEqual(2, len(fsList))
      fsList.normalize()
      self.failUnlessEqual(2, len(fsList))
      self.failUnless("one" in fsList)
      self.failUnless("two" in fsList)

   def testNormalize_004(self):
      """
      Test with a list containing two entries, with duplicates.
      """
      fsList = FilesystemList()
      fsList.append("one");
      fsList.append("one");
      self.failUnlessEqual(2, len(fsList))
      fsList.normalize()
      self.failUnlessEqual(1, len(fsList))
      self.failUnless("one" in fsList)

   def testNormalize_005(self):
      """
      Test with a list containing many entries, no duplicates.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      fsList.normalize()
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testNormalize_006(self):
      """
      Test with a list containing many entries, with duplicates.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(44, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      fsList.normalize()
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)


   ################
   # Test verify()
   ################
         
   def testVerify_001(self):
      """
      Test with an empty list.
      """
      fsList = FilesystemList()
      ok = fsList.verify()
      self.failUnlessEqual(True, ok)

   def testVerify_002(self):
      """
      Test with a non-empty list containing only invalid entries.
      """
      self.extractTar("tree9")
      fsList = FilesystemList()
      fsList.append(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(4, len(fsList))
      self.failUnless(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(False, ok)
      self.failUnlessEqual(4, len(fsList))
      self.failUnless(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]) in fsList)

   def testVerify_003(self):
      """
      Test with a non-empty list containing only valid entries (files only).
      """
      self.extractTar("tree1")
      path = self.buildPath(["tree1"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(True, ok)
      self.failUnlessEqual(8, len(fsList))
      self.failUnless(self.buildPath([ "tree1", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree1", "file007", ]) in fsList)

   def testVerify_004(self):
      """
      Test with a non-empty list containing only valid entries (directories
      only).
      """
      self.extractTar("tree2")
      path = self.buildPath(["tree2"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(11, count)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(True, ok)
      self.failUnlessEqual(11, len(fsList))
      self.failUnless(self.buildPath([ "tree2", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree2", "dir010", ]) in fsList)

   def testVerify_005(self):
      """
      Test with a non-empty list containing only valid entries (files and
      directories).
      """
      self.extractTar("tree4")
      path = self.buildPath(["tree4"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(81, count)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(True, ok)
      self.failUnlessEqual(81, len(fsList))
      self.failUnless(self.buildPath([ "tree4", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir001", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir002", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir003", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir004", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "dir008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir005", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "dir005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file007", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file008", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file009", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "dir006", "file010", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file005", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file006", ]) in fsList)
      self.failUnless(self.buildPath([ "tree4", "file007", ]) in fsList)

   def testVerify_006(self):
      """
      Test with a non-empty list containing only valid entries (files,
      directories and links).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(True, ok)
      self.failUnlessEqual(22, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testVerify_007(self):
      """
      Test with a non-empty list containing valid and invalid entries (files,
      directories and links).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(22, count)
      self.failUnlessEqual(22, len(fsList))
      fsList.append(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]))     # file won't exist on disk
      fsList.append(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(26, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(False, ok)
      self.failUnlessEqual(26, len(fsList))
      self.failUnless(self.buildPath([ "tree9", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-1" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-2" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-3" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "%s-4" % INVALID_FILE, ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fsList)

   def testVerify_008(self):
      """
      Test with a non-empty list containing valid and invalid entries (some
      containing spaces).
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      fsList = FilesystemList()
      count = fsList.addDirContents(path)
      self.failUnlessEqual(16, count)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)
      ok = fsList.verify()
      self.failUnlessEqual(True, ok)
      self.failUnlessEqual(16, len(fsList))
      self.failUnless(self.buildPath([ "tree11", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fsList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fsList)


###########################
# TestBackupFileList class
###########################

class TestBackupFileList(unittest.TestCase):

   """Tests for the BackupFileList class."""

   ################
   # Setup methods
   ################

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.resources = findResources(RESOURCES, DATA_DIRS)
      except Exception, e:
         self.fail(e)

   def tearDown(self):
      try:
         removedir(self.tmpdir)
      except: pass


   ##################
   # Utility methods
   ##################

   def extractTar(self, tarname):
      """Extracts a tarfile with a particular name."""
      extractTar(self.tmpdir, self.resources['%s.tar.gz' % tarname])

   def buildPath(self, components):
      """Builds a complete search path from a list of components."""
      components.insert(0, self.tmpdir)
      return buildPath(components)

   def buildRandomPath(self, maxlength, extension):
      """Builds a complete, randomly-named search path."""
      maxlength -= len(self.tmpdir)
      components = [ self.tmpdir, randomFilename(maxlength, extension), ]
      return buildPath(components)


   ################
   # Test addDir()
   ################
         
   def testAddDir_001(self):
      """
      Test that function is overridden, no exclusions.

      Since this function calls the superclass by definition, we can
      skimp a bit on validation and only ensure that it seems to be
      overridden properly.
      """
      self.extractTar("tree5")
      backupList = BackupFileList()
      dirPath = self.buildPath(["tree5", "dir001"])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))
      dirPath = self.buildPath(["tree5", "dir002", "link001", ])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([dirPath], backupList)

   def testAddDir_002(self):
      """
      Test that function is overridden, excludeFiles set.

      Since this function calls the superclass by definition, we can
      skimp a bit on validation and only ensure that it seems to be
      overridden properly.
      """
      self.extractTar("tree5")
      backupList = BackupFileList()
      backupList.excludeFiles = True
      dirPath = self.buildPath(["tree5", "dir001"])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))
      dirPath = self.buildPath(["tree5", "dir002", "link001", ])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([dirPath], backupList)

   def testAddDir_003(self):
      """
      Test that function is overridden, excludeDirs set.

      Since this function calls the superclass by definition, we can
      skimp a bit on validation and only ensure that it seems to be
      overridden properly.
      """
      self.extractTar("tree5")
      backupList = BackupFileList()
      backupList.excludeDirs = True
      dirPath = self.buildPath(["tree5", "dir001"])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))
      dirPath = self.buildPath(["tree5", "dir002", "link001", ])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))

   def testAddDir_004(self):
      """
      Test that function is overridden, excludeLinks set.

      Since this function calls the superclass by definition, we can
      skimp a bit on validation and only ensure that it seems to be
      overridden properly.
      """
      self.extractTar("tree5")
      backupList = BackupFileList()
      backupList.excludeLinks = True
      dirPath = self.buildPath(["tree5", "dir001"])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))
      dirPath = self.buildPath(["tree5", "dir002", "link001", ])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))

   def testAddDir_005(self):
      """
      Test that function is overridden, excludePaths set.

      Since this function calls the superclass by definition, we can
      skimp a bit on validation and only ensure that it seems to be
      overridden properly.
      """
      self.extractTar("tree5")
      backupList = BackupFileList()
      backupList.excludePaths = [ NOMATCH_PATH ]
      dirPath = self.buildPath(["tree5", "dir001"])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))
      dirPath = self.buildPath(["tree5", "dir002", "link001", ])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([dirPath], backupList)

   def testAddDir_006(self):
      """
      Test that function is overridden, excludePatterns set.

      Since this function calls the superclass by definition, we can
      skimp a bit on validation and only ensure that it seems to be
      overridden properly.
      """
      self.extractTar("tree5")
      backupList = BackupFileList()
      backupList.excludePaths = [ NOMATCH_PATH ]
      dirPath = self.buildPath(["tree5", "dir001"])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(0, len(backupList))
      dirPath = self.buildPath(["tree5", "dir002", "link001", ])
      count = backupList.addDir(dirPath)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([dirPath], backupList)


   ###################
   # Test totalSize()
   ###################
         
   def testTotalSize_001(self):
      """
      Test on an empty list.
      """
      backupList = BackupFileList()
      size = backupList.totalSize()
      self.failUnlessEqual(0, size)

   def testTotalSize_002(self):
      """
      Test on a non-empty list containing only valid entries.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      size = backupList.totalSize()
      self.failUnlessEqual(1116, size)

   def testTotalSize_004(self):
      """
      Test on a non-empty list (some containing spaces).
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(13, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in backupList)
      size = backupList.totalSize()
      self.failUnlessEqual(1085, size)

   def testTotalSize_005(self):
      """
      Test on a non-empty list containing a directory (which shouldn't be
      possible).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", "dir001", ]))     # back-door around addDir() 
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001" ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      size = backupList.totalSize()
      self.failUnlessEqual(1116, size)

   def testTotalSize_006(self):
      """
      Test on a non-empty list containing a non-existent file.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      size = backupList.totalSize()
      self.failUnlessEqual(1116, size)


   #########################
   # Test generateSizeMap()
   #########################
         
   def testGenerateSizeMap_001(self):
      """
      Test on an empty list.
      """
      backupList = BackupFileList()
      sizeMap = backupList.generateSizeMap()
      self.failUnlessEqual(0, len(sizeMap))

   def testGenerateSizeMap_002(self):
      """
      Test on a non-empty list containing only valid entries.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      sizeMap = backupList.generateSizeMap()
      self.failUnlessEqual(15, len(sizeMap))
      self.failUnlessEqual(243, sizeMap[self.buildPath([ "tree9", "dir001", "file001", ]) ])
      self.failUnlessEqual(268, sizeMap[self.buildPath([ "tree9", "dir001", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link003", ]) ])
      self.failUnlessEqual(134, sizeMap[self.buildPath([ "tree9", "dir002", "file001", ]) ])
      self.failUnlessEqual(74, sizeMap[self.buildPath([ "tree9", "dir002", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link003", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link004", ]) ])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree9", "file001", ]) ])
      self.failUnlessEqual(242, sizeMap[self.buildPath([ "tree9", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "link002", ]) ])

   def testGenerateSizeMap_004(self):
      """
      Test on a non-empty list (some containing spaces).
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(13, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in backupList)
      sizeMap = backupList.generateSizeMap()
      self.failUnlessEqual(13, len(sizeMap))
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "file001", ])])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "file with spaces", ])])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree11", "link001", ])])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree11", "link002", ])])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree11", "link003", ])])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree11", "link with spaces", ])])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "dir002", "file001", ])])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "dir002", "file002", ])])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "dir002", "file003", ])])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "dir with spaces", "file001", ])])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree11", "dir with spaces", "file with spaces", ])])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree11", "dir with spaces", "link002", ])])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree11", "dir with spaces", "link with spaces", ])])

   def testGenerateSizeMap_005(self):
      """
      Test on a non-empty list containing a directory (which shouldn't be
      possible).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", "dir001", ]))     # back-door around addDir() 
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001" ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      sizeMap = backupList.generateSizeMap()
      self.failUnlessEqual(15, len(sizeMap))
      self.failUnlessEqual(243, sizeMap[self.buildPath([ "tree9", "dir001", "file001", ]) ])
      self.failUnlessEqual(268, sizeMap[self.buildPath([ "tree9", "dir001", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link003", ]) ])
      self.failUnlessEqual(134, sizeMap[self.buildPath([ "tree9", "dir002", "file001", ]) ])
      self.failUnlessEqual(74, sizeMap[self.buildPath([ "tree9", "dir002", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link003", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link004", ]) ])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree9", "file001", ]) ])
      self.failUnlessEqual(242, sizeMap[self.buildPath([ "tree9", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "link002", ]) ])

   def testGenerateSizeMap_006(self):
      """
      Test on a non-empty list containing a non-existent file.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      sizeMap = backupList.generateSizeMap()
      self.failUnlessEqual(15, len(sizeMap))
      self.failUnlessEqual(243, sizeMap[self.buildPath([ "tree9", "dir001", "file001", ]) ])
      self.failUnlessEqual(268, sizeMap[self.buildPath([ "tree9", "dir001", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir001", "link003", ]) ])
      self.failUnlessEqual(134, sizeMap[self.buildPath([ "tree9", "dir002", "file001", ]) ])
      self.failUnlessEqual(74, sizeMap[self.buildPath([ "tree9", "dir002", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link003", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "dir002", "link004", ]) ])
      self.failUnlessEqual(155, sizeMap[self.buildPath([ "tree9", "file001", ]) ])
      self.failUnlessEqual(242, sizeMap[self.buildPath([ "tree9", "file002", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "link001", ]) ])
      self.failUnlessEqual(0, sizeMap[self.buildPath([ "tree9", "link002", ]) ])


   ###########################
   # Test generateDigestMap()
   ###########################
         
   def testGenerateDigestMap_001(self):
      """
      Test on an empty list.
      """
      backupList = BackupFileList()
      digestMap = backupList.generateDigestMap()
      self.failUnlessEqual(0, len(digestMap))

   def testGenerateDigestMap_002(self):
      """
      Test on a non-empty list containing only valid entries.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      digestMap = backupList.generateDigestMap()
      self.failUnlessEqual(6, len(digestMap))
      self.failUnlessEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[self.buildPath([ "tree9", "dir001", "file001", ])])
      self.failUnlessEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[self.buildPath([ "tree9", "dir001", "file002", ])])
      self.failUnlessEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[self.buildPath([ "tree9", "dir002", "file001", ])])
      self.failUnlessEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[self.buildPath([ "tree9", "dir002", "file002", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree9", "file001", ])])
      self.failUnlessEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[self.buildPath([ "tree9", "file002", ])])

   def testGenerateDigestMap_003(self):
      """
      Test on a non-empty list containing only valid entries (some containing spaces).
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(13, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in backupList)
      digestMap = backupList.generateDigestMap()
      self.failUnlessEqual(7, len(digestMap))
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "file001", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "file with spaces", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "dir002", "file001",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "dir002", "file002",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "dir002", "file003",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "dir with spaces", "file001",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree11", "dir with spaces", "file with spaces",])])

   def testGenerateDigestMap_004(self):
      """
      Test on a non-empty list containing a directory (which shouldn't be
      possible).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", "dir001", ]))     # back-door around addDir() 
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001" ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      digestMap = backupList.generateDigestMap()
      self.failUnlessEqual(6, len(digestMap))
      self.failUnlessEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[self.buildPath([ "tree9", "dir001", "file001", ])])
      self.failUnlessEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[self.buildPath([ "tree9", "dir001", "file002", ])])
      self.failUnlessEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[self.buildPath([ "tree9", "dir002", "file001", ])])
      self.failUnlessEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[self.buildPath([ "tree9", "dir002", "file002", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree9", "file001", ])])
      self.failUnlessEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[self.buildPath([ "tree9", "file002", ])])

   def testGenerateDigestMap_005(self):
      """
      Test on a non-empty list containing a non-existent file.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      digestMap = backupList.generateDigestMap()
      self.failUnlessEqual(6, len(digestMap))
      self.failUnlessEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[self.buildPath([ "tree9", "dir001", "file001", ])])
      self.failUnlessEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[self.buildPath([ "tree9", "dir001", "file002", ])])
      self.failUnlessEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[self.buildPath([ "tree9", "dir002", "file001", ])])
      self.failUnlessEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[self.buildPath([ "tree9", "dir002", "file002", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[self.buildPath([ "tree9", "file001", ])])
      self.failUnlessEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[self.buildPath([ "tree9", "file002", ])])

   def testGenerateDigestMap_006(self):
      """
      Test on an empty list, passing stripPrefix not None.
      """
      backupList = BackupFileList()
      prefix = "whatever"
      digestMap = backupList.generateDigestMap(stripPrefix=prefix)
      self.failUnlessEqual(0, len(digestMap))

   def testGenerateDigestMap_007(self):
      """
      Test on a non-empty list containing only valid entries, passing
      stripPrefix not None.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      prefix = normalizeDir(self.buildPath(["tree9", ]))
      digestMap = backupList.generateDigestMap(stripPrefix=prefix)
      self.failUnlessEqual(6, len(digestMap))
      self.failUnlessEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[buildPath([ "/", "dir001", "file001", ])])
      self.failUnlessEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[buildPath([ "/", "dir001", "file002", ])])
      self.failUnlessEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[buildPath([ "/", "dir002", "file001", ])])
      self.failUnlessEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[buildPath([ "/", "dir002", "file002", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "file001", ])])
      self.failUnlessEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[buildPath([ "/", "file002", ])])

   def testGenerateDigestMap_008(self):
      """
      Test on a non-empty list containing only valid entries (some containing
      spaces), passing stripPrefix not None.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(13, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in backupList)
      prefix = normalizeDir(self.buildPath(["tree11", ]))
      digestMap = backupList.generateDigestMap(stripPrefix=prefix)
      self.failUnlessEqual(7, len(digestMap))
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "file001", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "file with spaces", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "dir002", "file001",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "dir002", "file002",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "dir002", "file003",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "dir with spaces", "file001",])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "dir with spaces", "file with spaces",])])

   def testGenerateDigestMap_009(self):
      """
      Test on a non-empty list containing a directory (which shouldn't be
      possible), passing stripPrefix not None.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", "dir001", ]))     # back-door around addDir() 
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001" ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      prefix = normalizeDir(self.buildPath(["tree9", ]))
      digestMap = backupList.generateDigestMap(stripPrefix=prefix)
      self.failUnlessEqual(6, len(digestMap))
      self.failUnlessEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[buildPath([ "/", "dir001", "file001", ])])
      self.failUnlessEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[buildPath([ "/", "dir001", "file002", ])])
      self.failUnlessEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[buildPath([ "/", "dir002", "file001", ])])
      self.failUnlessEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[buildPath([ "/", "dir002", "file002", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "file001", ])])
      self.failUnlessEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[buildPath([ "/", "file002", ])])

   def testGenerateDigestMap_010(self):
      """
      Test on a non-empty list containing a non-existent file, passing
      stripPrefix not None.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      prefix = normalizeDir(self.buildPath(["tree9", ]))
      digestMap = backupList.generateDigestMap(stripPrefix=prefix)
      self.failUnlessEqual(6, len(digestMap))
      self.failUnlessEqual("4ff529531c7e897cd3df90ed76355de7e21e77ee", digestMap[buildPath([ "/", "dir001", "file001", ])])
      self.failUnlessEqual("9d473094a22ecf2ae299c25932c941795d1d6cba", digestMap[buildPath([ "/", "dir001", "file002", ])])
      self.failUnlessEqual("2f68cdda26b643ca0e53be6348ae1255b8786c4b", digestMap[buildPath([ "/", "dir002", "file001", ])])
      self.failUnlessEqual("0cc03b3014d1ca7188264677cf01f015d72d26cb", digestMap[buildPath([ "/", "dir002", "file002", ])])
      self.failUnlessEqual("3ef0b16a6237af9200b7a46c1987d6a555973847", digestMap[buildPath([ "/", "file001", ])])
      self.failUnlessEqual("fae89085ee97b57ccefa7e30346c573bb0a769db", digestMap[buildPath([ "/", "file002", ])])


   ########################
   # Test generateFitted()
   ########################
         
   def testGenerateFitted_001(self):
      """
      Test on an empty list.
      """
      backupList = BackupFileList()
      fittedList = backupList.generateFitted(2000)
      self.failUnlessEqual(0, len(fittedList))

   def testGenerateFitted_002(self):
      """
      Test on a non-empty list containing only valid entries, all of which fit.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      fittedList = backupList.generateFitted(2000)
      self.failUnlessEqual(15, len(fittedList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fittedList)

   def testGenerateFitted_003(self):
      """
      Test on a non-empty list containing only valid entries (some containing
      spaces), all of which fit.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(13, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in backupList)
      fittedList = backupList.generateFitted(2000)
      self.failUnlessEqual(13, len(fittedList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in fittedList)

   def testGenerateFitted_004(self):
      """
      Test on a non-empty list containing only valid entries, some of which fit.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      fittedList = backupList.generateFitted(80)
      self.failUnlessEqual(10, len(fittedList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fittedList)

   def testGenerateFitted_005(self):
      """
      Test on a non-empty list containing only valid entries, none of which fit.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      fittedList = backupList.generateFitted(0)
      self.failUnlessEqual(0, len(fittedList))
      fittedList = backupList.generateFitted(50)
      self.failUnlessEqual(9, len(fittedList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fittedList)

   def testGenerateFitted_006(self):
      """
      Test on a non-empty list containing a directory (which shouldn't be
      possible).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", "dir001", ]))     # back-door around addDir() 
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001" ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      fittedList = backupList.generateFitted(2000)
      self.failUnlessEqual(15, len(fittedList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fittedList)

   def testGenerateFitted_007(self):
      """
      Test on a non-empty list containing a non-existent file.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      fittedList = backupList.generateFitted(2000)
      self.failUnlessEqual(15, len(fittedList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in fittedList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in fittedList)


   #########################
   # Test generateTarfile()
   #########################
         
   def testGenerateTarfile_001(self):
      """
      Test on an empty list.
      """
      backupList = BackupFileList()
      tarPath = self.buildPath(["file.tar", ])      
      self.failUnlessRaises(ValueError, backupList.generateTarfile, tarPath)
      self.failUnless(not os.path.exists(tarPath))

   def testGenerateTarfile_002(self):
      """
      Test on a non-empty list containing a directory (which shouldn't be
      possible).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", "dir001", ]))     # back-door around addDir() 
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001" ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar",])      
      backupList.generateTarfile(tarPath)
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(16, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001/" ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_003(self):
      """
      Test on a non-empty list containing a non-existent file, ignore=False.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar", ])      
      self.failUnlessRaises(tarfile.TarError, backupList.generateTarfile, tarPath, ignore=False)
      self.failUnless(not os.path.exists(tarPath))

   def testGenerateTarfile_004(self):
      """
      Test on a non-empty list containing a non-existent file, ignore=True.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      backupList.append(self.buildPath([ "tree9", INVALID_FILE, ]))     # file won't exist on disk
      self.failUnlessEqual(16, len(backupList))
      self.failUnless(self.buildPath([ "tree9", INVALID_FILE ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar", ])      
      backupList.generateTarfile(tarPath, ignore=True)
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_005(self):
      """
      Test on a non-empty list containing only valid entries, with an invalid mode.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar", ])      
      self.failUnlessRaises(ValueError, backupList.generateTarfile, tarPath, mode="bogus")
      self.failUnless(not os.path.exists(tarPath))

   def testGenerateTarfile_006(self):
      """
      Test on a non-empty list containing only valid entries, default mode.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar", ])      
      backupList.generateTarfile(tarPath)
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_007(self):
      """
      Test on a non-empty list (some containing spaces), default mode.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(13, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in backupList)
      tarPath = self.buildPath(["file.tar", ])      
      backupList.generateTarfile(tarPath)
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(13, len(tarList))
      self.failUnless(self.buildPath([ "tree11", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ])[1:] in tarList)

   def testGenerateTarfile_008(self):
      """
      Test on a non-empty list containing only valid entries, 'tar' mode.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar", ])      
      backupList.generateTarfile(tarPath)
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_009(self):
      """
      Test on a non-empty list containing only valid entries, 'targz' mode.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar.gz", ])      
      backupList.generateTarfile(tarPath, mode="targz")
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_010(self):
      """
      Test on a non-empty list containing only valid entries, 'tarbz2' mode.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildPath(["file.tar.bz2", ])      
      backupList.generateTarfile(tarPath, mode="tarbz2")
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_011(self):
      """
      Test on a non-empty list containing only valid entries, 'tar' mode, long target name.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildRandomPath(260, ".tar")
      backupList.generateTarfile(tarPath, mode="tar")
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_012(self):
      """
      Test on a non-empty list containing only valid entries, 'targz' mode, long target name.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildRandomPath(260, ".tar.gz")
      backupList.generateTarfile(tarPath, mode="targz")
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)

   def testGenerateTarfile_013(self):
      """
      Test on a non-empty list containing only valid entries, 'tarbz2' mode, long target name.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      tarPath = self.buildRandomPath(260, ".tar.bz2")
      backupList.generateTarfile(tarPath, mode="tarbz2")
      self.failUnless(tarfile.is_tarfile(tarPath))
      tarFile = tarfile.open(tarPath)
      tarList = tarFile.getnames()
      tarFile.close()
      self.failUnlessEqual(15, len(tarList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "file002", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link001", ])[1:] in tarList)
      self.failUnless(self.buildPath([ "tree9", "link002", ])[1:] in tarList)


   #########################
   # Test removeUnchanged()
   #########################
         
   def testRemoveUnchanged_001(self):
      """
      Test on an empty list with an empty digest map.
      """
      digestMap = {}
      backupList = BackupFileList()
      self.failUnlessEqual(0, len(backupList))
      backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(0, len(backupList))

   def testRemoveUnchanged_002(self):
      """
      Test on an empty list with an non-empty digest map.
      """
      digestMap = { self.buildPath([ "tree9", "dir001", "file001", ]):"4ff529531c7e897cd3df90ed76355de7e21e77ee", 
                    self.buildPath([ "tree9", "dir001", "file002", ]):"9d473094a22ecf2ae299c25932c941795d1d6cba", 
                    self.buildPath([ "tree9", "dir002", "file001", ]):"2f68cdda26b643ca0e53be6348ae1255b8786c4b", 
                    self.buildPath([ "tree9", "dir002", "file002", ]):"0cc03b3014d1ca7188264677cf01f015d72d26cb", 
                    self.buildPath([ "tree9", "file001", ])          :"3ef0b16a6237af9200b7a46c1987d6a555973847", 
                    self.buildPath([ "tree9", "file002", ])          :"fae89085ee97b57ccefa7e30346c573bb0a769db", }
      backupList = BackupFileList()
      self.failUnlessEqual(0, len(backupList))
      backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(0, len(backupList))

   def testRemoveUnchanged_003(self):
      """
      Test on an non-empty list with an empty digest map.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)

   def testRemoveUnchanged_004(self):
      """
      Test with a digest map containing only entries that are not in the list.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { self.buildPath([ "tree9", "dir003", "file001", ]):"4ff529531c7e897cd3df90ed76355de7e21e77ee", 
                    self.buildPath([ "tree9", "dir003", "file002", ]):"9d473094a22ecf2ae299c25932c941795d1d6cba", 
                    self.buildPath([ "tree9", "dir004", "file001", ]):"2f68cdda26b643ca0e53be6348ae1255b8786c4b", 
                    self.buildPath([ "tree9", "dir004", "file002", ]):"0cc03b3014d1ca7188264677cf01f015d72d26cb", 
                    self.buildPath([ "tree9", "file003", ])          :"3ef0b16a6237af9200b7a46c1987d6a555973847", 
                    self.buildPath([ "tree9", "file004", ])          :"fae89085ee97b57ccefa7e30346c573bb0a769db", }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)

   def testRemoveUnchanged_005(self):
      """
      Test with a digest map containing only entries that are in the list, with
      non-matching digests.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { self.buildPath([ "tree9", "dir001", "file001", ]):"4ff529531c7e8AAAAAAAAAAAAAAAAAA7e21e77ee", 
                    self.buildPath([ "tree9", "dir001", "file002", ]):"9d473094a22ecAAAAAAAAAAAAAAAAAA95d1d6cba", 
                    self.buildPath([ "tree9", "dir002", "file001", ]):"2f68cdda26b64AAAAAAAAAAAAAAAAAA5b8786c4b", 
                    self.buildPath([ "tree9", "dir002", "file002", ]):"0cc03b3014d1cAAAAAAAAAAAAAAAAAA5d72d26cb", 
                    self.buildPath([ "tree9", "file001", ])          :"3ef0b16a6237aAAAAAAAAAAAAAAAAAA555973847", 
                    self.buildPath([ "tree9", "file002", ])          :"fae89085ee97bAAAAAAAAAAAAAAAAAAbb0a769db", }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)

   def testRemoveUnchanged_006(self):
      """
      Test with a digest map containing only entries that are in the list, with
      matching digests.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { self.buildPath([ "tree9", "dir001", "file001", ]):"4ff529531c7e897cd3df90ed76355de7e21e77ee", 
                    self.buildPath([ "tree9", "dir001", "file002", ]):"9d473094a22ecf2ae299c25932c941795d1d6cba", 
                    self.buildPath([ "tree9", "dir002", "file001", ]):"2f68cdda26b643ca0e53be6348ae1255b8786c4b", 
                    self.buildPath([ "tree9", "dir002", "file002", ]):"0cc03b3014d1ca7188264677cf01f015d72d26cb", 
                    self.buildPath([ "tree9", "file001", ])          :"3ef0b16a6237af9200b7a46c1987d6a555973847", 
                    self.buildPath([ "tree9", "file002", ])          :"fae89085ee97b57ccefa7e30346c573bb0a769db", }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(9, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)

   def testRemoveUnchanged_007(self):
      """
      Test with a digest map containing both entries that are and are not in
      the list, with non-matching digests.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { self.buildPath([ "tree9", "dir001", "file001", ]):"4ff529531cCCCCCCCCCCCCCCCCCCCCCCCCCe77ee", 
                    self.buildPath([ "tree9", "dir001", "file002", ]):"9d473094a2CCCCCCCCCCCCCCCCCCCCCCCCCd6cba", 
                    self.buildPath([ "tree9", "dir003", "file001", ]):"2f68cdda26CCCCCCCCCCCCCCCCCCCCCCCCC86c4b", 
                    self.buildPath([ "tree9", "dir003", "file002", ]):"0cc03b3014CCCCCCCCCCCCCCCCCCCCCCCCCd26cb", 
                    self.buildPath([ "tree9", "file001", ])          :"3ef0b16a62CCCCCCCCCCCCCCCCCCCCCCCCC73847", 
                    self.buildPath([ "tree9", "file003", ])          :"fae89085eeCCCCCCCCCCCCCCCCCCCCCCCCC769db", }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)

   def testRemoveUnchanged_008(self):
      """
      Test with a digest map containing both entries that are and are not in
      the list, with matching digests.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { self.buildPath([ "tree9", "dir001", "file001", ]):"4ff529531c7e897cd3df90ed76355de7e21e77ee", 
                    self.buildPath([ "tree9", "dir001", "file002", ]):"9d473094a22ecf2ae299c25932c941795d1d6cba", 
                    self.buildPath([ "tree9", "dir003", "file001", ]):"2f68cdda26b643ca0e53be6348ae1255b8786c4b", 
                    self.buildPath([ "tree9", "dir003", "file002", ]):"0cc03b3014d1ca7188264677cf01f015d72d26cb", 
                    self.buildPath([ "tree9", "file001", ])          :"3ef0b16a6237af9200b7a46c1987d6a555973847", 
                    self.buildPath([ "tree9", "file003", ])          :"fae89085ee97b57ccefa7e30346c573bb0a769db", }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(12, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)

   def testRemoveUnchanged_009(self):
      """
      Test with a digest map containing both entries that are and are not in
      the list, with matching and non-matching digests.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9"])
      digestMap = { self.buildPath([ "tree9", "dir001", "file001", ]):"4ff529531AAAAAAAAAAAAAAAAAAAAAAAe21e77ee", 
                    self.buildPath([ "tree9", "dir001", "file002", ]):"9d473094a22ecf2ae299c25932c941795d1d6cba", 
                    self.buildPath([ "tree9", "dir003", "file001", ]):"2f68cdda26b643ca0e53be6348ae1255b8786c4b", 
                    self.buildPath([ "tree9", "dir003", "file002", ]):"0cc03b3014d1ca7188264677cf01f015d72d26cb", 
                    self.buildPath([ "tree9", "file001", ])          :"3ef0b16a6237af9200b7a46c1987d6a555973847", 
                    self.buildPath([ "tree9", "file003", ])          :"fae89085ee97b57ccefa7e30346c573bb0a769db", }
      backupList = BackupFileList()
      count = backupList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)
      count = backupList.removeUnchanged(digestMap)
      self.failUnless(isinstance(backupList, BackupFileList))  # make sure we just replaced it
      self.failUnlessEqual(2, count)
      self.failUnlessEqual(13, len(backupList))
      self.failUnless(self.buildPath([ "tree9", "dir001", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir001", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link003", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "file002", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in backupList)
      self.failUnless(self.buildPath([ "tree9", "link002", ]) in backupList)


   #########################
   # Test _generateDigest()
   #########################
         
   def testGenerateDigest_001(self):
      """
      Test that _generateDigest gives back same result as the slower simplistic
      implementation for a set of files (just using all of the resource files).
      """
      for key in self.resources.keys():
         path = self.resources[key]
         digest1 = sha.new(open(path).read()).hexdigest()
         digest2 = BackupFileList._generateDigest(path)
         self.failUnlessEqual(digest1, digest2, "Digest for %s varies: [%s] vs [%s]." % (path, digest1, digest2))


##########################
# TestPurgeItemList class
##########################

class TestPurgeItemList(unittest.TestCase):

   """Tests for the PurgeItemList class."""

   ################
   # Setup methods
   ################

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.resources = findResources(RESOURCES, DATA_DIRS)
      except Exception, e:
         self.fail(e)

   def tearDown(self):
      try:
         removedir(self.tmpdir)
      except: pass


   ##################
   # Utility methods
   ##################

   def extractTar(self, tarname):
      """Extracts a tarfile with a particular name."""
      extractTar(self.tmpdir, self.resources['%s.tar.gz' % tarname])

   def buildPath(self, components):
      """Builds a complete search path from a list of components."""
      components.insert(0, self.tmpdir)
      return buildPath(components)


   ########################
   # Test addDirContents()
   ########################
         
   def testAddDirContents_001(self):
      """
      Attempt to add a directory that doesn't exist; no exclusions.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_002(self):
      """
      Attempt to add a file; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_003(self):
      """
      Attempt to add a soft link; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      count = purgeList.addDir(path)
      self.failUnlessEqual(1, count)
      self.failUnlessEqual([path], purgeList)

   def testAddDirContents_004(self):
      """
      Attempt to add an empty directory containing ignore file; no exclusions.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_005(self):
      """
      Attempt to add an empty directory; no exclusions.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_006(self):
      """
      Attempt to add an non-empty directory containing ignore file; no
      exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_007(self):
      """
      Attempt to add an non-empty directory; no exclusions.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(7, len(purgeList))
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in purgeList)

   def testAddDirContents_008(self):
      """
      Attempt to add a directory that doesn't exist; excludeFiles set.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_009(self):
      """
      Attempt to add a file; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_010(self):
      """
      Attempt to add a soft link; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_011(self):
      """
      Attempt to add an empty directory containing ignore file; excludeFiles set.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludeFiles = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_012(self):
      """
      Attempt to add an empty directory; excludeFiles set.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_013(self):
      """
      Attempt to add an non-empty directory containing ignore file; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludeFiles = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_014(self):
      """
      Attempt to add an non-empty directory; excludeFiles set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual(4, len(purgeList))
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in purgeList)

   def testAddDirContents_015(self):
      """
      Attempt to add a directory that doesn't exist; excludeDirs set.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_016(self):
      """
      Attempt to add a file; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_017(self):
      """
      Attempt to add a soft link; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_018(self):
      """
      Attempt to add an empty directory containing ignore file; excludeDirs set.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludeDirs = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_019(self):
      """
      Attempt to add an empty directory; excludeDirs set.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_020(self):
      """
      Attempt to add an non-empty directory containing ignore file; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludeDirs = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_021(self):
      """
      Attempt to add an non-empty directory; excludeDirs set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(3, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in purgeList)

   def testAddDirContents_023(self):
      """
      Attempt to add a directory that doesn't exist; excludeLinks set.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_024(self):
      """
      Attempt to add a file; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_025(self):
      """
      Attempt to add a soft link; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_026(self):
      """
      Attempt to add an empty directory containing ignore file; excludeLinks set.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludeLinks = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_027(self):
      """
      Attempt to add an empty directory; excludeLinks set.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_028(self):
      """
      Attempt to add an non-empty directory containing ignore file; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludeLinks = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_029(self):
      """
      Attempt to add an non-empty directory; excludeLinks set.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(6, count)
      self.failUnlessEqual(6, len(purgeList))
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in purgeList)

   def testAddDirContents_030(self):
      """
      Attempt to add a directory that doesn't exist; with excludePaths
      including the path.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_031(self):
      """
      Attempt to add a file; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_032(self):
      """
      Attempt to add a soft link; with excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ path ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_033(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePaths including the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePaths = [ path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_034(self):
      """
      Attempt to add an empty directory; with excludePaths including the path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_035(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePaths including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePaths = [ path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_036(self):
      """
      Attempt to add an non-empty directory; with excludePaths including the
      main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_037(self):
      """
      Attempt to add a directory that doesn't exist; with excludePaths not
      including the path.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_038(self):
      """
      Attempt to add a file; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_039(self):
      """
      Attempt to add a soft link; with excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_040(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePaths not including the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePaths = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_041(self):
      """
      Attempt to add an empty directory; with excludePaths not including the
      path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_042(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePaths not including the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePaths = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_043(self):
      """
      Attempt to add an non-empty directory; with excludePaths not including
      the main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(7, len(purgeList))
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in purgeList)

   def testAddDirContents_044(self):
      """
      Attempt to add a directory that doesn't exist; with excludePatterns
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_045(self):
      """
      Attempt to add a file; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_046(self):
      """
      Attempt to add a soft link; with excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_047(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePatterns matching the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_048(self):
      """
      Attempt to add an empty directory; with excludePatterns matching the
      path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_049(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePatterns matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_050(self):
      """
      Attempt to add an non-empty directory; with excludePatterns matching the
      main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*%s.*" % path ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_051(self):
      """
      Attempt to add a directory that doesn't exist; with excludePatterns not
      matching the path.
      """
      path = self.buildPath([INVALID_FILE])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_052(self):
      """
      Attempt to add a file; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "file001"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_053(self):
      """
      Attempt to add a soft link; with excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "link001"])     # link to a file
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir002", "link001"])  # link to a dir
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_054(self):
      """
      Attempt to add an empty directory containing ignore file; with
      excludePatterns not matching the path.
      """
      self.extractTar("tree7")
      path = self.buildPath(["tree7", "dir001"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_055(self):
      """
      Attempt to add an empty directory; with excludePatterns not matching the
      path.
      """
      self.extractTar("tree8")
      path = self.buildPath(["tree8", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_056(self):
      """
      Attempt to add an non-empty directory containing ignore file; with
      excludePatterns not matching the path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir008"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testAddDirContents_057(self):
      """
      Attempt to add an non-empty directory; with excludePatterns not matching
      the main directory path.
      """
      self.extractTar("tree5")
      path = self.buildPath(["tree5", "dir001"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ NOMATCH_PATH ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(7, count)
      self.failUnlessEqual(7, len(purgeList))
      self.failUnless(self.buildPath(["tree5", "dir001", "dir001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir003",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "dir004",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file001",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "file002",]) in purgeList)
      self.failUnless(self.buildPath(["tree5", "dir001", "link001",]) in purgeList)

   def testAddDirContents_058(self):
      """
      Attempt to add a large tree with no exclusions.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(135, count)
      self.failUnlessEqual(135, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in purgeList)

   def testAddDirContents_059(self):
      """
      Attempt to add a large tree, with excludeFiles set.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      purgeList.excludeFiles = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(41, count)
      self.failUnlessEqual(41, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in purgeList)

   def testAddDirContents_060(self):
      """
      Attempt to add a large tree, with excludeDirs set.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      purgeList.excludeDirs = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(94, count)
      self.failUnlessEqual(94, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in purgeList)

   def testAddDirContents_061(self):
      """
      Attempt to add a large tree, with excludeLinks set.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      purgeList.excludeLinks = True
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(95, count)
      self.failUnlessEqual(95, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in purgeList)

   def testAddDirContents_062(self):
      """
      Attempt to add a large tree, with excludePaths set to exclude some entries.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      purgeList.excludePaths = [ self.buildPath([ "tree6", "dir001", "dir002", ]), 
                              self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]), 
                              self.buildPath([ "tree6", "dir003", "dir002", "file001", ]), 
                              self.buildPath([ "tree6", "dir003", "dir002", "file002", ]), ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(124, count)
      self.failUnlessEqual(124, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in purgeList)

   def testAddDirContents_063(self):
      """
      Attempt to add a large tree, with excludePatterns set to exclude some entries.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      purgeList.excludePatterns = [ ".*file001.*", ".*tree6\/dir002\/dir001.*" ]
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(107, count)
      self.failUnlessEqual(107, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "ignore", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir003", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in purgeList)

   def testAddDirContents_064(self):
      """
      Attempt to add a large tree, with ignoreFile set to exclude some directories.
      """
      self.extractTar("tree6")
      path = self.buildPath(["tree6"])
      purgeList = PurgeItemList()
      purgeList.ignoreFile = "ignore"
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(78, count)
      self.failUnlessEqual(78, len(purgeList))
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "file009", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir001", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "file008", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file006", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "file007", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "dir003", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link004", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "dir002", "link005", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree6", "link002", ]) in purgeList)

   def testAddDirContents_065(self):
      """
      Attempt to add a link to a file.
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9", "dir002", "link003", ])
      purgeList = PurgeItemList()
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_066(self):
      """
      Attempt to add a link to a directory (which should add its contents).
      """
      self.extractTar("tree9")
      path = self.buildPath(["tree9", "link002" ])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(8, count)
      self.failUnlessEqual(8, len(purgeList))
      self.failUnless(self.buildPath([ "tree9", "link002", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "link002", "link004", ]) in purgeList)

   def testAddDirContents_067(self):
      """
      Attempt to add an invalid link (i.e. a link that points to something that
      doesn't exist).
      """
      self.extractTar("tree10")
      path = self.buildPath(["tree10", "link001"])
      purgeList = PurgeItemList()
      self.failUnlessRaises(ValueError, purgeList.addDirContents, path)

   def testAddDirContents_068(self):
      """
      Attempt to add directory containing an invalid link (i.e. a link that
      points to something that doesn't exist).
      """
      self.extractTar("tree10")
      path = self.buildPath(["tree10"])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(2, count)
      self.failUnlessEqual(2, len(purgeList))
      self.failUnless(self.buildPath([ "tree10", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree10", "dir002", ]) in purgeList)

   def testAddDirContents_069(self):
      """
      Attempt to add a directory containing items with spaces.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", ])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(15, count)
      self.failUnlessEqual(15, len(purgeList))
      self.failUnless(self.buildPath([ "tree11", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "file with spaces", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "link003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "link with spaces", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir002", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in purgeList)

   def testAddDirContents_070(self):
      """
      Attempt to add a directory which has a name containing spaces.
      """
      self.extractTar("tree11")
      path = self.buildPath(["tree11", "dir with spaces", ])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual(4, len(purgeList))
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "file with spaces", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree11", "dir with spaces", "link with spaces", ]) in purgeList)

   def testAddDirContents_071(self):
      """
      Attempt to add a directory which has a UTF-8 filename in it.
      """
      self.extractTar("tree12")
      path = self.buildPath(["tree12", "unicode", ])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(5, count)
      self.failUnlessEqual(5, len(purgeList))
      self.failUnless(self.buildPath([ "tree12", "unicode", "README.strange-name", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "utflist.long.gz", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "utflist.cp437.gz", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "utflist.short.gz", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree12", "unicode", "\xe2\x99\xaa\xe2\x99\xac", ]) in purgeList)

   def testAddDirContents_072(self):
      """
      Attempt to add a directory which has several UTF-8 filenames in it.
      This test data was taken from Rick Lowe's problems around the release of v1.10.
      """
      self.extractTar("tree13")
      path = self.buildPath(["tree13", ])
      purgeList = PurgeItemList()
      count = purgeList.addDirContents(path)
      self.failUnlessEqual(10, count)
      self.failUnlessEqual(10, len(purgeList))
      self.failUnless(self.buildPath([ "tree13", "Les mouvements de r\x82forme.doc", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "l'\x82nonc\x82.sxw", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard - renvois et bibliographie.sxw", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard copie finale.sxw", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard de vinci - page titre.sxw", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "l\x82onard de vinci.sxw", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "Rammstein - B\x81ck Dich.mp3", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "megaherz - Glas Und Tr\x84nen.mp3", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "Megaherz - Mistst\x81ck.MP3", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree13", "Rammstein - Mutter - B\x94se.mp3", ]) in purgeList)


   ####################
   # Test removeAged()
   ####################

   def testRemoveYoungFiles_001(self):
      """
      Test on an empty list, daysOld < 0.
      """
      daysOld = -1
      purgeList = PurgeItemList()
      self.failUnlessRaises(ValueError, purgeList.removeYoungFiles, daysOld)

   def testRemoveYoungFiles_002(self):
      """
      Test on a non-empty list, daysOld < 0.
      """
      daysOld = -1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree1", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      self.failUnlessRaises(ValueError, purgeList.removeYoungFiles, daysOld)

   def testRemoveYoungFiles_003(self):
      """
      Test on an empty list, daysOld = 0
      """
      daysOld = 0
      purgeList = PurgeItemList()
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_004(self):
      """
      Test on a non-empty list containing only directories, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree2")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree2", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir001", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir002", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree2", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in purgeList)

   def testRemoveYoungFiles_005(self):
      """
      Test on a non-empty list containing only links, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree9")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree9", "link001", ]))
      purgeList.addDir(self.buildPath([ "tree9", "dir002", "link001", ]))
      purgeList.addFile(self.buildPath([ "tree9", "dir002", "link004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in purgeList)

   def testRemoveYoungFiles_006(self):
      """
      Test on a non-empty list containing only non-existent files, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.append(self.buildPath([ "tree1", "stuff001", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff002", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff003", ]))     # append, since it doesn't exist on disk
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree1", "stuff001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff003", ]) in purgeList)

   def testRemoveYoungFiles_007(self):
      """
      Test on a non-empty list containing existing files "touched" to current time, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]))
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_008(self):
      """
      Test on a non-empty list containing existing files "touched" to being 1 hour old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_1_HOUR)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_1_HOUR)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_009(self):
      """
      Test on a non-empty list containing existing files "touched" to being 2 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_2_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_2_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_010(self):
      """
      Test on a non-empty list containing existing files "touched" to being 12 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_12_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_12_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_011(self):
      """
      Test on a non-empty list containing existing files "touched" to being 23 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_23_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_23_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_012(self):
      """
      Test on a non-empty list containing existing files "touched" to being 24 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_24_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_24_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_013(self):
      """
      Test on a non-empty list containing existing files "touched" to being 25 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_25_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_25_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_014(self):
      """
      Test on a non-empty list containing existing files "touched" to being 47 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_47_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_47_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_015(self):
      """
      Test on a non-empty list containing existing files "touched" to being 48 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_48_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_48_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_016(self):
      """
      Test on a non-empty list containing existing files "touched" to being 49 hours old, daysOld = 0.
      """
      daysOld = 0
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_49_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_49_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file003", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_017(self):
      """
      Test on an empty list, daysOld = 1
      """
      daysOld = 1
      purgeList = PurgeItemList()
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_018(self):
      """
      Test on a non-empty list containing only directories, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree2")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree2", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir001", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir002", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree2", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in purgeList)

   def testRemoveYoungFiles_019(self):
      """
      Test on a non-empty list containing only links, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree9")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree9", "link001", ]))
      purgeList.addDir(self.buildPath([ "tree9", "dir002", "link001", ]))
      purgeList.addFile(self.buildPath([ "tree9", "dir002", "link004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in purgeList)

   def testRemoveYoungFiles_020(self):
      """
      Test on a non-empty list containing only non-existent files, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.append(self.buildPath([ "tree1", "stuff001", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff002", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff003", ]))     # append, since it doesn't exist on disk
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree1", "stuff001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff003", ]) in purgeList)

   def testRemoveYoungFiles_021(self):
      """
      Test on a non-empty list containing existing files "touched" to current time, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]))
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_022(self):
      """
      Test on a non-empty list containing existing files "touched" to being 1 hour old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_1_HOUR)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_1_HOUR)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_023(self):
      """
      Test on a non-empty list containing existing files "touched" to being 2 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_2_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_2_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_024(self):
      """
      Test on a non-empty list containing existing files "touched" to being 12 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_12_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_12_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_025(self):
      """
      Test on a non-empty list containing existing files "touched" to being 23 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_23_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_23_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_026(self):
      """
      Test on a non-empty list containing existing files "touched" to being 24 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_24_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_24_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(2, len(purgeList))
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_027(self):
      """
      Test on a non-empty list containing existing files "touched" to being 25 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_25_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_25_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_028(self):
      """
      Test on a non-empty list containing existing files "touched" to being 47 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_47_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_47_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_029(self):
      """
      Test on a non-empty list containing existing files "touched" to being 48 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_48_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_48_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_030(self):
      """
      Test on a non-empty list containing existing files "touched" to being 49 hours old, daysOld = 1.
      """
      daysOld = 1
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_49_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_49_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_031(self):
      """
      Test on an empty list, daysOld = 2
      """
      daysOld = 2
      purgeList = PurgeItemList()
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_032(self):
      """
      Test on a non-empty list containing only directories, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree2")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree2", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir001", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir002", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree2", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in purgeList)

   def testRemoveYoungFiles_033(self):
      """
      Test on a non-empty list containing only links, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree9")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree9", "link001", ]))
      purgeList.addDir(self.buildPath([ "tree9", "dir002", "link001", ]))
      purgeList.addFile(self.buildPath([ "tree9", "dir002", "link004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in purgeList)

   def testRemoveYoungFiles_034(self):
      """
      Test on a non-empty list containing only non-existent files, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.append(self.buildPath([ "tree1", "stuff001", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff002", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff003", ]))     # append, since it doesn't exist on disk
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree1", "stuff001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff003", ]) in purgeList)

   def testRemoveYoungFiles_035(self):
      """
      Test on a non-empty list containing existing files "touched" to current time, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]))
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_036(self):
      """
      Test on a non-empty list containing existing files "touched" to being 1 hour old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_1_HOUR)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_1_HOUR)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_037(self):
      """
      Test on a non-empty list containing existing files "touched" to being 2 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_2_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_2_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_038(self):
      """
      Test on a non-empty list containing existing files "touched" to being 12 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_12_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_12_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_039(self):
      """
      Test on a non-empty list containing existing files "touched" to being 23 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_23_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_23_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_040(self):
      """
      Test on a non-empty list containing existing files "touched" to being 24 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_24_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_24_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_041(self):
      """
      Test on a non-empty list containing existing files "touched" to being 25 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_25_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_25_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_042(self):
      """
      Test on a non-empty list containing existing files "touched" to being 47 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_47_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_47_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_043(self):
      """
      Test on a non-empty list containing existing files "touched" to being 48 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_48_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_48_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_044(self):
      """
      Test on a non-empty list containing existing files "touched" to being 49 hours old, daysOld = 2.
      """
      daysOld = 2
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_49_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_49_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(2, count)
      self.failUnless(self.buildPath([ "tree1", "file001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "file004", ]) in purgeList)

   def testRemoveYoungFiles_045(self):
      """
      Test on an empty list, daysOld = 3
      """
      daysOld = 3
      purgeList = PurgeItemList()
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_046(self):
      """
      Test on a non-empty list containing only directories, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree2")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree2", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir001", ]))
      purgeList.addDir(self.buildPath([ "tree2", "dir002", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree2", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree2", "dir002", ]) in purgeList)

   def testRemoveYoungFiles_047(self):
      """
      Test on a non-empty list containing only links, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree9")
      purgeList = PurgeItemList()
      purgeList.addDir(self.buildPath([ "tree9", "link001", ]))
      purgeList.addDir(self.buildPath([ "tree9", "dir002", "link001", ]))
      purgeList.addFile(self.buildPath([ "tree9", "dir002", "link004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree9", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree9", "dir002", "link004", ]) in purgeList)

   def testRemoveYoungFiles_048(self):
      """
      Test on a non-empty list containing only non-existent files, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.append(self.buildPath([ "tree1", "stuff001", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff002", ]))     # append, since it doesn't exist on disk
      purgeList.append(self.buildPath([ "tree1", "stuff003", ]))     # append, since it doesn't exist on disk
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(0, count)
      self.failUnlessEqual(3, len(purgeList))
      self.failUnless(self.buildPath([ "tree1", "stuff001", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff002", ]) in purgeList)
      self.failUnless(self.buildPath([ "tree1", "stuff003", ]) in purgeList)

   def testRemoveYoungFiles_049(self):
      """
      Test on a non-empty list containing existing files "touched" to current time, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]))
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]))
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_050(self):
      """
      Test on a non-empty list containing existing files "touched" to being 1 hour old, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_1_HOUR)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_1_HOUR)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_051(self):
      """
      Test on a non-empty list containing existing files "touched" to being 2 hours old, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_2_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_2_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_052(self):
      """
      Test on a non-empty list containing existing files "touched" to being 12 hours old, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_12_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_12_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_053(self):
      """
      Test on a non-empty list containing existing files "touched" to being 23 hours old, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addFile(self.buildPath([ "tree1", "file001", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file002", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file003", ]))
      purgeList.addFile(self.buildPath([ "tree1", "file004", ]))
      changeFileAge(self.buildPath([ "tree1", "file001", ]), AGE_23_HOURS)
      changeFileAge(self.buildPath([ "tree1", "file002", ]))
      changeFileAge(self.buildPath([ "tree1", "file003", ]))
      changeFileAge(self.buildPath([ "tree1", "file004", ]), AGE_23_HOURS)
      count = purgeList.removeYoungFiles(daysOld)
      self.failUnlessEqual(4, count)
      self.failUnlessEqual([], purgeList)

   def testRemoveYoungFiles_054(self):
      """
      Test on a non-empty list containing existing files "touched" to being 24 hours old, daysOld = 3.
      """
      daysOld = 3
      self.extractTar("tree1")
      purgeList = PurgeItemList()
      purgeList.addF