# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#                                                                   #
#              C E D A R                                            #
#          S O L U T I O N S       "Software done right."           #
#           S O F T W A R E                                         #
#                                                                   #
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#                                                                   #
# Copyright (c) 2003 Kenneth J. Pronovici.                          #
# All rights reserved.                                              #
#                                                                   #
# This program is free software; you can redistribute it and/or     #
# modify it under the terms of the GNU General Public License,      #
# Version 2, as published by the Free Software Foundation.          #
#                                                                   #
# This program is distributed in the hope that it will be useful,   #
# but WITHOUT ANY WARRANTY; without even the implied warranty of    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              #
#                                                                   #
# Copies of the GNU General Public License are available from       #
# the Free Software Foundation website, http://www.gnu.org/.        #
#                                                                   #
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.2)
# Project  : WordUtils
# Revision : $Id: tree.py,v 1.18 2003/07/16 05:23:04 pronovic Exp $
# Purpose  : Implementation of a ternary search tree
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# This file was created with a width of 132 characters, and NO tabs.

######################
# Pydoc documentation
######################

"""
Implementation of a ternary search tree and related functionality.

What is a ternary search tree?
==============================

   What is a ternary search tree?  The article U{Ternary Search Trees
   <http://www.ddj.com/documents/s=921/ddj9804a/9804a.htm>} by Jon Bentley and
   Bob Sedgewick in Doctor Dobb's Journal, April 1998, discusses them.  The
   article has this to say::

      When you have to store a set of strings, what data structure should
      you use? You could use hash tables, which sprinkle the strings
      throughout an array. Access is fast, but information about relative
      order is lost. Another option is the use of binary search trees,
      which store strings in order, and are fairly fast. Or you could use
      digital search tries, which are lightning fast, but use lots of
      space.

      ...[T]ernary search trees...combine the time efficiency of digital
      tries with the space efficiency of binary search trees. The
      resulting structure is faster than hashing for many typical search
      problems, and supports a broader range of useful problems and
      operations. Ternary searches are faster than hashing and more
      powerful, too.


Implementation Notes
====================

   The implementation below is conceptually based on the data structure
   described in the Ternary Search Trees article, although I have made some
   changes because the article's implementation is focused toward C.  The
   Python objects are undoubtedly larger than their C counterparts, but there
   is some code simplificiation due to Python's more sophisticated data
   structures.  The Python implementation is likely also somewhat slower than
   the C implementation, although I do not have any benchmarks.

   Most of the public methods in the TernarySearchTree object are implemented
   in what I think of as hierarchical fashion.  What this means is, a public
   method will be implemented almost entirely in terms of other, private
   methods.  Since the TernarySearchTree object supports both on-disk and
   in-memory trees, for instance, most of the public methods are implemented in
   terms of two different private methods, where the public method doesn't do
   much except choose between them.    

   On first glace through this code, it may appear obvious that I should just
   create two different objects (an in-memory tree and an on-disk tree) to
   clean things up.  Actually, since there is very little in-memory structure
   for an on-disk tree (just a filename, really) there would be very little
   benefit to reorganizing the code.  As it stands, it's not too hard to
   follow.  

   I think the method naming convention should be obvious, but I'll spell it
   out anyway.  Single-word methods are lower-case only, i.e. C{lower()}.
   Multiple-word methods are C{lowerUpper()}.  The in-memory variant of a
   method starts with C{_mem}, i.e. C{_mem_lowerUpper()}, while the on-disk
   variant starts with C{_disk}, i.e. C{_disk_lowerUpper()}.  A recursively
   called method ends with C{_r}, i.e. C{_mem_lowerUpper_r()}.  For the most
   part, any sub-methods associated with a public method are grouped with it,
   unless they stand on their own.  

   Note that C{ON_DISK} trees are inherently read-only (i.e., you cannot insert
   new words into or remove words from C{ON_DISK} trees).  They're also not
   going to be as fast as C{IN_MEMORY} trees.  Use them when you want to
   conserve memory and when performance isn't your top priority.  

   The L{TernarySearchTree.compress} method exists for the TernarySearchTree
   object.  However, it is a no-op right now.  I have provided it (along with
   its regression tests) so that it will be easy to support compression in the
   future, if I can implement a decent compression algorithm.  


Database Format
===============

   The L{TernarySearchTree.save} method provides the option to store a tree on
   disk in either a wordlist or "database" format.  The "database" format is
   essentially an on-disk representation of the in-memory tree, using file
   pointers instead of object references.  Once a database has been saved, you
   have the option of using it to load an in-memory tree, or instead accessing
   the database directly from disk, using very little memory in the process.

   The database file format is not intended to be human readable.  However, it
   should be portable across operating systems, since I use the Python marshal
   module to write the data to disk.  As of now, I have not done extensive
   testing on platforms other than Linux.


Documentation Notes
===================

   I only provide docstrings for public interface methods.  I figure that the
   arguments to the private methods should be fairly obvious given the
   documentation for the public methods.

   Keep in mind that any backslash characters in Epydoc markup must be escaped,
   so in the markup you'll always see two of them together.  This is sometimes
   a bit confusing. 

@author: Kenneth J. Pronovici <pronovic@ieee.org>
@version: $Revision: 1.18 $
"""


########################################################################
# Imported modules
########################################################################

import sys
try:
   import os
   import shutil
   import marshal
except ImportError, e:
    print "Failed to import standard modules: %s" % e
    print "Please try setting the PYTHONPATH environment variable."
    sys.exit(1)


#######################################################################
# Module-wide configuration and constants
#######################################################################

# Definitions for the type of tree we're working with
IN_MEMORY = 0
ON_DISK   = 1


#######################################################################
# Classes
#######################################################################

##################
# TreeError class
##################

class TreeError(Exception):
   """General exception class for this module."""
   def __init__(self, value):
      Exception.__init__(self, value)
      self.value = value
   def __str__(self):
      return self.value


#############
# Node class
#############

class Node:

   ######################
   # Class documentation
   ######################

   """Class representing a node in an in-memory ternary search tree."""


   #####################
   # Instance variables
   #####################

   splitchar = None
   endpoint = False
   lokid = None
   eqkid = None
   hikid = None
   
   
   ####################
   # __init__() method
   ####################

   def __init__ (self, key=None):
      """
      Default constructor.
      @param key: key to be used to construct this node
      @type key: single-character string     
      """
      if key is not None:
         self.splitchar = key[0].lower()
      else:
         self.splitchar = None
      self.endpoint = False
      self.lokid = None
      self.eqkid = None
      self.hikid = None


##########################
# TernarySearchTree class
##########################

class TernarySearchTree:


   ######################
   # Class documentation
   ######################

   """Class representing a ternary search tree."""


   #####################
   # Instance variables
   #####################

   root = None        # root node of tree
   type = None        # IN_MEMORY or ON_DISK
   db = None          # location of database on disk
   wl = None          # location of wordlist on disk


   ####################
   # __init__() method
   ####################

   def __init__(self, type=IN_MEMORY, db=None, wl=None):
      """
      Default constructor.

      See the L{loadFile} method for details on how the C{db} and
      C{wl} arguments are used.

      @param type: type of tree to be created
      @type type: one of C{[IN_MEMORY, ON_DISK]}

      @param db: database on disk to be used to create tree
      @type db: valid filesytem path as a string

      @param wl: wordlist on disk to be used to create tree
      @type wl: valid filesystem path as a string

      @raise TreeError: under exception circumstances
      """  
      if type not in [IN_MEMORY, ON_DISK]:
         raise TreeError, "Tree type must be either IN_MEMORY or ON_DISK."
      self.type = type
      self.loadFile(db, wl)


   ####################
   # loadFile() method
   ####################

   def loadFile(self, db=None, wl=None):
      """
      Reads a wordlist or database file into a tree.

      Either a database or a wordlist may be used, but not both.  If a wordlist
      is used, it is assumed to be properly sorted on disk (i.e. the words must
      be in alphabetical order).  Neither file will ever be written to.

      Calling this function on a tree that has been previously initialized will
      overwrite the contents of that tree.

      You may not choose to load a wordlist into an C{ON_DISK} tree.  First,
      create an C{IN_MEMORY} tree using the wordlist.  Then, save the
      C{IN_MEMORY} tree as a database.  Then, you can load use the database on
      disk as the source for your C{ON_DISK} tree.

      @param db: database on disk to be used to initialize tree
      @type db: valid filesytem path as a string

      @param wl: wordlist on disk to be used to initialize tree
      @type wl: valid filesystem path as a string

      @raise TreeError: under exception circumstances
      """  
      if db is not None and wl is not None:
         raise TreeError, "Object may be initialized with either database or wordlist, not both."
      elif db is not None or wl is not None:
         oldroot = self.root
         olddb = self.db
         oldwl = self.wl
         self.root = None
         self.db = None
         self.wl = None
         try:
            if self.type == IN_MEMORY:
               if db is not None:
                  self.db = db
                  self.root = self._mem_loadFileDatabase(self.db)
               else:
                  self.wl = wl
                  self.root = self._mem_loadFileWordlist(self.wl)
            elif self.type == ON_DISK:
               if db is not None:
                  if not os.path.exists(db):
                     raise IOError
                  self.db = db
               else:
                  raise TreeError, "On-disk tree must be loaded from database."
         except Exception, detail:
            # If this process fails, make sure to reset the state
            self.root = oldroot
            self.db = olddb
            self.wl = oldwl
            raise TreeError, "%s" % detail

   def _mem_loadFileWordlist(self, wl):
      node = None
      for line in open(wl).xreadlines():   # assumes that wordlist is sorted properly
         node = self._insert(node, line[:-1].lower())
      return node

   def _mem_loadFileDatabase(self, db):
      file = open(db, "r+b")
      try:
         head = marshal.load(file)
         return self._mem_loadFileDatabase_r(None, file, head)
      except ValueError, detail:
         raise TreeError, "Unable to load (%s): %s - maybe not a database?" % (db, detail)

   def _mem_loadFileDatabase_r(self, node, file, fp):
      if node is None:
         node = Node()
      if fp != -1:
         file.seek(fp)
         node.splitchar = chr(marshal.load(file))
         if node.splitchar == -1:
            node.splitchar = None
         if marshal.load(file) == 0:
            node.endpoint = False
         else:
            node.endpoint = True
         lofp = marshal.load(file)
         eqfp = marshal.load(file)
         hifp = marshal.load(file)
         if lofp != -1:
            node.lokid = self._mem_loadFileDatabase_r(node.lokid, file, lofp)
         if eqfp != -1:
            node.eqkid = self._mem_loadFileDatabase_r(node.eqkid, file, eqfp)
         if hifp != -1:
            node.hikid = self._mem_loadFileDatabase_r(node.hikid, file, hifp)
      return node


   ####################
   # loadList() method
   ####################

   def loadList(self, list):
      """
      Reads a Python list into a tree.

      Calling this function on a tree that has been previously initialized will
      overwrite the contents of that tree.

      You may only load a list into an C{IN_MEMORY} tree.  If you need the list
      in an on-disk tree, create an C{IN_MEMORY} tree using the list.  Then,
      save the C{IN_MEMORY} tree as a database.  Then, you can load the database
      on disk as the source for your C{ON_DISK} tree.

      @param list: Python list to be used to initialize tree
      @type list: list of strings

      @raise TreeError: under exception circumstances
      """    
      oldroot = self.root
      olddb = self.db
      oldwl = self.wl
      self.root = None
      self.db = None
      self.wl = None
      try:
         if self.type == IN_MEMORY:
            self.root = self._mem_loadList(list)
         elif self.type == ON_DISK:
            raise TreeError, "On-disk tree must be loaded from database."
      except Exception, detail:
         # If this process fails, make sure to reset the state
         self.root = oldroot
         self.db = olddb
         self.wl = oldwl
         raise TreeError, "%s" % detail

   def _mem_loadList(self, list):
      node = None
      for word in list:
         node = self._insert(node, word.lower())
      return node


   ################
   # list() method
   ################

   def list(self):
      """
      Returns a sorted list representation of the tree.
      @return: sorted list of words in the tree
      @raise DawgError: under exception circumstances
      """   
      list = []
      func = lambda x: list.append(x)
      self.traverse(func)
      list.sort()
      return list


   ####################
   # compress() method
   ####################
 
   def compress(self, ratio=False):
      """
      Compresses a tree, optionally returning compression ratio.

      This method may only be used on C{IN_MEMORY} trees.

      I{This method has not yet been implemented, and exists only
      as a stub.  The method and its regression tests are in place
      to support future functionality, if I can every come up with
      a way to make compression work.}

      @return:
            - compression relative to original, as a percentage,
              if C{ratio} is True
            - C{None} if C{ratio} is False

      @raise TreeError: under exception circumstances 
      """
      if self.type == ON_DISK:
         raise TreeError, "On-disk trees may not be compressed. To compress, create in-memory database from db file."
      if ratio:
         return 100.0
      else:
         return None


   ####################
   # traverse() method
   ####################

   def traverse(self, func=None):
      """
      Traverses a tree and executes C{func(word)} for each word.

      If no function is provided, then the default function is::

         lambda x: sys.stdout.write("%s\\n" % x)

      which will print out each word on its own line (the words will be
      unordered).

      @param func: function to be called when an endpoint is reached
      @type func: reference to a function which takes one argument, a word

      @raise DawgError: under exception circumstances 
      """
      if func is None:
         func = lambda x: sys.stdout.write("%s\n" % x)
      if self.type == IN_MEMORY:
         self._mem_traverse(self.root, "", func)
      else:
         self._disk_traverse(self.db, "", func)

   def _mem_traverse(self, node, word, func):
      if node is not None:
         if node.splitchar is not None:
            word += node.splitchar
         if node.endpoint:
            func(word)
         if node.lokid is not None:
            self._mem_traverse(node.lokid, word[1:], func)
         if node.eqkid is not None:
            self._mem_traverse(node.eqkid, word, func)
         if node.hikid is not None:
            self._mem_traverse(node.hikid, word[:-1], func)

   def _disk_traverse(self, filepath, word, func):
      if filepath is not None:
         if os.path.exists(filepath) and os.stat(filepath)[6] > 0:
            file = open(filepath, "r+b")
            file.seek(0)
            try:
               head = marshal.load(file)
               self._disk_traverse_r(file, head, word, func)
            except ValueError, detail:
               raise TreeError, "Unable to load (%s): %s - maybe not a database?" % (filepath, detail)

   def _disk_traverse_r(self, file, fp, word, func):
      if fp != -1:
         file.seek(fp)
         splitchar = chr(marshal.load(file))
         if splitchar != -1:
            word += splitchar
         if marshal.load(file) != 0:         # endpoint
            func(word)
         lofp = marshal.load(file)
         eqfp = marshal.load(file)
         hifp = marshal.load(file)
         if lofp != -1:
            self._disk_traverse_r(file, lofp, word[1:], func)
         if eqfp != -1:
            self._disk_traverse_r(file, eqfp, word, func)
         if hifp != -1:
            self._disk_traverse_r(file, hifp, word[:-1], func)


   ################
   # save() method
   ################

   def save(self, db=None, wl=None, overwrite=False):
      """
      Writes a tree to disk as a wordlist and/or in a portable "database"
      format.

      Both the C{wl} and C{db} arguments may be provided.  This will
      cause the method to write both a wordlist and a database to disk.

      Note: if the tree was created as C{ON_DISK}, then this method cannot be
      used to overwrite the in-use database on disk.

      @param db: database to be written to on disk
      @type db: valid filesytem path as a string

      @param wl: wordlist to be written to on disk
      @type wl: valid filesystem path as a string

      @param overwrite: indicates whether to overwrite an existing file on disk
      @type overwrite: boolean C{True}/C{False}

      @raise TreeError: under exception circumstances
      """ 
      if wl is not None:
         self._saveWordlist(wl, overwrite)
      if db is not None:
         self._saveDatabase(self.root, self.db, db, overwrite)
      
   def _saveWordlist(self, newwl, overwrite):
      if os.path.exists(newwl) and not overwrite:
         raise TreeError, "Wordlist file (%s) exists but overwrite not specified." % newwl
      list = self.list()
      if len(list) == 0:
         open(newwl, "w").write("")
      else:
         fp = open(newwl, "w")
         for word in self.list():
            fp.write("%s\n" % word)

   def _saveDatabase(self, root, db, newdb, overwrite):
      if db == newdb:
         raise TreeError, "Object's current database (%s) may not be overwritten by save." % db
      if os.path.exists(newdb) and not overwrite:
         raise TreeError, "Database file (%s) exists but overwrite not specified." % newdb
      if self.type == IN_MEMORY:
         self._mem_saveDatabase(root, newdb)
      else:
         self._disk_saveDatabase(db, newdb)

   def _mem_saveDatabase(self, node, newdb):
      file = open(newdb, "w+b")
      marshal.dump(int(0), file)
      head = self._mem_saveDatabase_r(node, file)
      file.seek(0)
      marshal.dump(int(head), file)

   def _mem_saveDatabase_r(self, node, file):
      nodefp = -1
      if node is not None:
         lofp = -1
         eqfp = -1
         hifp = -1
         if node.lokid is None:
            lofp = self._mem_saveDatabase_r(node.lokid, file)
         if node.eqkid is not None:
            eqfp = self._mem_saveDatabase_r(node.eqkid, file)
         if node.hikid is not None:
            hifp = self._mem_saveDatabase_r(node.hikid, file)
         nodefp = file.tell()
         if node.splitchar is None:
            marshal.dump(int(-1))
         else:
            marshal.dump(int(ord(node.splitchar)), file)
         marshal.dump(int(node.endpoint), file)
         marshal.dump(int(lofp), file)
         marshal.dump(int(eqfp), file)
         marshal.dump(int(hifp), file)
      return nodefp

   def _disk_saveDatabase(self, db, newdb):
      if db is None:
         self._mem_saveDatabase(None, newdb)
      else:
         shutil.copyfile(db, newdb)


   ##################
   # search() method
   ##################

   def search(self, key):
      """
      Searches for a complete string in a tree.

      @param key: string to search for
      @type key: string, of any length

      @return: boolean indicating whether the key was found in the tree.

      @raise TreeError: under exception circumstances
      """ 
      if self.type == IN_MEMORY:
         return self._mem_search(self.root, key.lower())
      else:
         return self._disk_search(self.db, key.lower())
   
   def _mem_search(self, node, key):
      i = 0
      while node is not None and i < len(key):
         if key[i] < node.splitchar:
            node = node.lokid
         elif key[i] == node.splitchar:
            if len(key[i:]) == 1 and node.endpoint:
               return True
            else:
               i += 1
               node = node.eqkid
         else:
            node = node.hikid
      return False

   def _disk_search(self, filepath, key):
      i = 0
      file = open(filepath, "r+b")
      file.seek(0)
      try:
         fp = marshal.load(file)
         while fp != -1 and i < len(key):
            file.seek(fp)
            splitchar = chr(marshal.load(file))
            endpoint = marshal.load(file)
            lofp = marshal.load(file)
            eqfp = marshal.load(file)
            hifp = marshal.load(file)
            if key[i] < splitchar:
               fp = lofp
            elif key[i] == splitchar:
               if len(key[i:]) == 1 and endpoint:
                  return True
               else:
                  i += 1
                  fp = eqfp
            else:
               fp = hifp
         return False
      except ValueError, detail:
         raise TreeError, "Unable to load (%s): %s - maybe not a database?" % (filepath, detail)


   #########################
   # patternSearch() method
   #########################
      
   def patternSearch(self, pattern):
      """
      Searches for a pattern in a tree.

      A pattern uses the C{.} (period) character to represent a wildcard.  So,
      for instance, the pattern C{'.ai.'} would match the words C{'rail'},
      C{'hail'}, C{'mail'}, etc. if those words were in the tree.

      @return: list of words in the tree which match the pattern

      @raise TreeError: under exception circumstances
      """
      if self.type == IN_MEMORY:
         return self._mem_patternSearch(self.root, pattern, "")
      else:
         return self._disk_patternSearch(self.db, pattern, "")
      
   def _mem_patternSearch(self, node, pattern, word):
      results = []
      if node is not None:
         word += node.splitchar
         if pattern[0] == "." or pattern[0] < node.splitchar:
            if node.lokid is not None:
               results += self._mem_patternSearch(node.lokid, pattern, word[1:])
         if pattern[0] == "." or pattern[0] == node.splitchar:
            if len(pattern) == 1:
               if node.endpoint: 
                  results.append(word)
            else:
               if node.eqkid is not None:
                  results += self._mem_patternSearch(node.eqkid, pattern[1:], word)
         if pattern[0] == "." or pattern[0] > node.splitchar:
            if node.hikid is not None:
               results += self._mem_patternSearch(node.hikid, pattern, word[:-1]);
      return results

   def _disk_patternSearch(self, filepath, pattern, word):
      file = open(filepath, "r+b")
      file.seek(0)
      try:
         fp = marshal.load(file)
         return self._disk_patternSearch_r(file, fp, pattern, word)
      except ValueError, detail:
         raise TreeError, "Unable to load (%s): %s - maybe not a database?" % (filepath, detail)
      
   def _disk_patternSearch_r(self, file, fp, pattern, word):
      results = []
      if fp != -1:
         file.seek(fp)
         splitchar = chr(marshal.load(file))
         endpoint = marshal.load(file)
         lofp = marshal.load(file)
         eqfp = marshal.load(file)
         hifp = marshal.load(file)
         word += splitchar
         if pattern[0] == "." or pattern[0] < splitchar:
            if lofp != -1:
               results += self._disk_patternSearch_r(file, lofp, pattern, word[1:])
         if pattern[0] == "." or pattern[0] == splitchar:
            if len(pattern) == 1:
               if endpoint:
                  results.append(word)
            else:
               if eqfp != -1:
                  results += self._disk_patternSearch_r(file, eqfp, pattern[1:], word)
         if pattern[0] == "." or pattern[0] > splitchar:
            if hifp != -1:
               results += self._disk_patternSearch_r(file, hifp, pattern, word[:-1])
      return results

      
   ###############
   # add() method
   ###############
  
   def add(self, key):
      """
      Adds a key (word) to an existing tree.

      This function inserts into a tree by building a list of the words in the
      tree, inserting the new key into the list, and then re-building the tree
      based on the new list.  Obviously, this is rather memory intensive and
      inefficient.  The whole point of using a tree in the first place is to
      avoid having a list of words around.  However, I haven't found any other
      good way to do this.

      You may not really need this functionality.  Normally, a tree would be
      filled using the L{loadFile} method.  This L{add} method would only be
      used to add a new word to an existing tree.  If you only need to add
      a few words to an existing tree, consider whether you might be better off
      keeping around a small Python list or dictionary to hold the extra word,
      instead of adding to the tree.

      @raise DawgError: under exception circumstances
      """ 
      if self.type == ON_DISK:
         raise TreeError, "On-disk trees are read-only.  To modify, create in-memory database from db file."
      list = self.list()
      list.append(key)
      list.sort()
      self.loadList(list)


   ##################
   # remove() method
   ##################

   def remove(self, key):
      """
      Removes a key (word) from a tree.

      This function removes a word from a tree by building a list of the words
      in the tree, removing the key from the list, and then re-building the
      tree based on the new list.  Obviously, this is rather memory intensive
      and inefficient.  The whole point of using a tree in the first place is
      to avoid having a list of words around.  However, I haven't found any
      other good way to do this.

      You may not really need this functionality.  If you only need to remove a
      few words from an existing tree, consider whether you might be better off
      just keeping around a small Python list or dictionary to hold those
      words.  You could then use the list or dictionary to validate the results
      returned from the L{patternSearch} method.

      @raise DawgError: under exception circumstances
      """ 
      if self.type == ON_DISK:
         raise TreeError, "On-disk trees are read-only.  To modify, create in-memory database from db file."
      list = self.list()
      list.remove(key)
      self.loadList(list)


   ###################
   # _insert() method
   ###################

   def _insert(self, node, key):
      if self.type == ON_DISK:
         raise TreeError, "On-disk trees are read-only.  To modify, create in-memory database from db file."
      return self._mem_insert(node, key)

   def _mem_insert(self, node, key):
      if node is None:
         node = Node(key)
      if key[0] < node.splitchar:
         node.lokid = self._mem_insert(node.lokid, key)
      elif key[0] == node.splitchar:
         if len(key) == 1:
            node.endpoint = True
         else:
            node.eqkid = self._mem_insert(node.eqkid, key[1:])
      else:
         node.hikid = self._mem_insert(node.hikid, key)
      return node


########################################################################
# Module entry point
########################################################################

# Ensures that the module isn't run if someone just imports it.
if __name__ == '__main__':
   print "Module just exists to be imported, sorry."

