#!/usr/bin/env python
# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#                                                                   #
#              C E D A R                                            #
#          S O L U T I O N S       "Software done right."           #
#           S O F T W A R E                                         #
#                                                                   #
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#                                                                   #
# Copyright (c) 2003 Kenneth J. Pronovici.                          #
# All rights reserved.                                              #
#                                                                   #
# This program is free software; you can redistribute it and/or     #
# modify it under the terms of the GNU General Public License,      #
# Version 2, as published by the Free Software Foundation.          #
#                                                                   #
# This program is distributed in the hope that it will be useful,   #
# but WITHOUT ANY WARRANTY; without even the implied warranty of    #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              #
#                                                                   #
# Copies of the GNU General Public License are available from       #
# the Free Software Foundation website, http://www.gnu.org/.        #
#                                                                   #
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.2)
# Project  : WordUtils
# Revision : $Id: treetests.py,v 1.13 2003/07/16 05:23:04 pronovic Exp $
# Purpose  : Unit tests for WordUtils/tree.py.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# This file was created with a width of 132 characters, and NO tabs.

########################################################################
# Notes
########################################################################

# Debugging in here is DAMN complicated.  If you have a test:
#
#        def test():
#           try:
#              # stuff
#           finally:
#              # remove files
#
# you may mysteriously have the 'stuff' fail, and you won't get any
# exceptions reported to you.  Best thing to do if you get strange
# situations like this is to move 'stuff' out of the try block - that
# will usually clear things up.
#
# Run pychecker against this code using the -l option to ignore 
# unused local variables - in many cases, local variables in here
# are not really "used", they're just created for testing purposes.


########################################################################
# Pydoc documentation
########################################################################

"""
Unit tests for WordUtils/tree.py.

Reference treetests.plan for the plan that this module implements.
"""

__author__  = "Kenneth J. Pronovici"
__version__ = "$Revision: 1.13 $"


########################################################################
# Import modules and do runtime validations
########################################################################

# Import standard modules
import sys
try:
   import os
   import xreadlines
   import tempfile
   import shutil
   import unittest
except ImportError, e:
    print "Failed to import standard modules: %s" % e
    print "Please try setting the PYTHONPATH environment variable."
    sys.exit(1)

# Try to find the right WordUtils module.  
# We want to make sure the tests use the modules in the current source
# tree, not any versions previously-installed elsewhere, if possible.
try:
   if os.path.exists(os.path.join(".", "WordUtils", "tree.py")):
      sys.path.insert(0, ".")
   elif os.path.basename(os.getcwd()) == "unittest" and os.path.exists(os.path.join("..", "WordUtils", "tree.py")):
      sys.path.insert(0, "..")
   else:
      print "WARNING: WordUtils modules were not found in the expected"
      print "location.  If the import succeeds, you may be using an"
      print "unexpected version of WordUtils."
      print ""
   from WordUtils.tree import TernarySearchTree
   from WordUtils.tree import IN_MEMORY
   from WordUtils.tree import ON_DISK
   from WordUtils.tree import TreeError
except ImportError, e:
   print "Failed to import WordUtils modules: %s" % e
   print ""
   print "You must either run the unit tests from the WordUtils source"
   print "tree, or properly set the PYTHONPATH enviroment variable."
   sys.exit(1)

# Check the Python version.  We require 2.2 or greater.
try:
   if map(int, [sys.version_info[0], sys.version_info[1]]) < [2, 2]:
      print "Python version 2.2 or greater required, sorry."
      sys.exit(1)
except:
   # sys.version_info isn't available before 2.0
   print "Python version 2.2 or greater required, sorry."
   sys.exit(1)


#######################################################################
# Module-wide configuration and constants
#######################################################################

# Words not in wordlist1
words_0_04a   = [ 'anecdotal', 'balfour', 'bolton', 'carcinogen', 'cochlea', 
                  'cowhide', 'desecrater', 'edmonds', 'extrusion', 'fringe', 
                  'growth', 'honky', 'iniquitous', 'knack', 'lumen', 'midget', 
                  'nightmarish', 'paregoric', 'podium', 'queer', 'rsvp', 'shaw', 
                  'spermatozoa', 'switchboard', 'tote', 'vest', 'xi' ]

# Words in wordlist1
words_0_04b   = [ 'alumnus', 'asteroid', 'bellatrix', 'brandenburg', 'caret',
                  'clasp', 'continuity', 'dabble', 'dilapidate', 'effusion',
                  'exhilarate', 'fluster', 'gestalt', 'handcuff', 'horrible',
                  'inexorable', 'journeymen', 'lengthwise', 'malpractice', 
                  'milton', 'nest', 'orville', 'perturbate', 'practitioner', 
                  'racy', 'ron', 'sedan', 'sleety', 'statuette', 'tabloid', 
                  'toni', 'urge', 'wet' ]
 
# Patterns not in wordlist1
patterns_0_05 = [ '.a.b.c', 'zz.al.', 'word...word', 'y..tta', 'bogus', 's.tuff' ] + words_0_04a

# Patterns in worldlist1
patterns_0_06 = { '.'        : [ 'a', 'k' ], 
                  '..'       : [ 'ad', 'el', 'ny', 'ph' ],
                  '...'      : [ 'bet', 'cdc', 'cia', 'cop', 'etc', 'fag', 'fee', 'her', 
                                 'imp', 'lim', 'mit', 'nay', 'ron', 'sat', 'spy', 'wet' ],
                  '.ac.'     : [ 'racy' ],
                  '.car.'    : [ 'scary' ],
                  '..r..'    : [ 'caret', 'curio', 'perez', 'surah', 'virgo' ],
                  'a...'     : [ 'able', 'aida', 'alex', 'aver', 'axis' ],
                  '..n'      : [ 'ron' ],
                  '.ee.'     : [ 'veer', 'week' ],
                  'b..e'     : [ 'byte' ],
                  't.......' : [ 'tanzania', 'tranquil', 'twilight' ],
                  '...a....' : [ 'alkaloid', 'almagest', 'escalate', 'fleawort', 'hilarity', 'strategy' ],
                  '.....c..' : [ 'pussycat', 'vertices' ],
                  '.....c'   : [ 'atypic', 'idetic', 'sextic' ],
                  'veer'     : [ 'veer' ] }


#######################################################################
# Test Case Classes
#######################################################################

##############
# Test1 class
##############

class Test1(unittest.TestCase):

   """Test 1: Loading and traversing trees; list representation."""

   def setUp(self):
      try:
         self.data = find_data()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      pass
         
   def test_1_01(self):
      tree = TernarySearchTree()
      self.failUnless(tree.type == IN_MEMORY)

   def test_1_02(self):
      tree = TernarySearchTree(type=IN_MEMORY)
      self.failUnless(tree.type == IN_MEMORY)

   def test_1_03(self):
      tree = TernarySearchTree(type=ON_DISK)
      self.failUnless(tree.type == ON_DISK)

   def test_1_04(self):
      try:
         tree = TernarySearchTree(type='bad')
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_05(self):
      try:
         tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'], db=self.data['wordlist1.tdb'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_06(self):
      try:
         tree = TernarySearchTree(type=ON_DISK, wl=self.data['wordlist1'], db=self.data['wordlist1.tdb'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_07(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      self.failUnless(tree.wl == self.data['wordlist1'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist1)

   def test_1_08(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_09(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_10(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_1_11(self):
      tree = TernarySearchTree(type=IN_MEMORY)
      try:
         tree.loadFile(wl=self.data['wordlist1'], db=self.data['wordlist1.tdb'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_12(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      self.failUnless(tree.wl == self.data['wordlist1'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist1)

   def test_1_13(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_14(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_15(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_1_16(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      root = tree.root
      wl = tree.wl
      db = tree.db
      try:
         tree.loadFile(wl=tempfile.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)
      try:
         tree.loadFile(db=tempfile.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)

   def test_1_17(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      root = tree.root
      wl = tree.wl
      db = tree.db
      try:
         tree.loadFile(wl=tempfile.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)
      try:
         tree.loadFile(db=tempfile.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)

   def test_1_18(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_1_19(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      root = tree.root
      wl = tree.wl
      db = tree.db
      try:
         tree.loadFile(wl=tempfile.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)
      try:
         tree.loadFile(db=tempfile.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)

   def test_1_20(self):
      wordlist2 = wordlist_as_list(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      tree.loadFile(wl=self.data['wordlist2'])
      self.failUnless(tree.wl == self.data['wordlist2'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist2)

   def test_1_21(self):
      wordlist2 = wordlist_as_list(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      tree.loadFile(db=self.data['wordlist2.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db  == self.data['wordlist2.tdb'])
      self.failUnless(tree.list() == wordlist2)

   def test_1_22(self):
      wordlist2 = wordlist_as_list(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      tree.loadFile(db=self.data['wordlist2.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db  == self.data['wordlist2.tdb'])
      self.failUnless(tree.list() == wordlist2)

   def test_1_23(self):
      wordlist2 = wordlist_as_list(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      tree.loadFile(wl=self.data['wordlist2'])
      self.failUnless(tree.wl == self.data['wordlist2'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist2)

   def test_1_24(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_1_25(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_1_26(self):
      wordlist2 = wordlist_as_list(self.data['wordlist2'])
      tree = TernarySearchTree(type=ON_DISK)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      tree.loadFile(db=self.data['wordlist2.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist2.tdb'])
      self.failUnless(tree.list() == wordlist2)

   def test_1_27(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_1_28(self):
      try:
         tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_29(self):
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      try:
         tree.compress()
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_30(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      self.failUnless(tree.list() == wordlist1)
      tree.compress()
      self.failUnless(tree.list() == wordlist1)

   def test_1_31(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK)
      try:
         tree.loadList(wordlist1)
      except TreeError: pass
      else: self.fail("Expected TreeError");

   def test_1_32(self):
      wordlist1 = wordlist_as_list(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadList(wordlist1)
      self.failUnless(tree.type == IN_MEMORY)
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db is None)
      list1 = tree.list()
      self.failUnless(list1 == wordlist1)
      tree.compress()
      list2 = tree.list()
      self.failUnless(list1 == list2)
      list2 = tree.list()
      self.failUnless(list1 == list2)


##############
# Test2 class
##############

class Test2(unittest.TestCase):

   """Test 2: Saving tree objects to disk."""

   def setUp(self):
      try:
         self.data = find_data()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      pass

   def test_2_01(self):
      try:
         tree = TernarySearchTree(type=IN_MEMORY)
         wl = tempfile.mktemp()
         db = tempfile.mktemp()
         tree.save(wl=wl, db=db)
         self.failUnless(os.path.exists(wl))
         self.failUnless(os.path.exists(db))
         self.failUnless(os.stat(wl)[6] == 0)
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(tree1.list() == [])
         self.failUnless(tree2.list() == [])
         self.failUnless(tree3.list() == [])
      finally:
         os.remove(wl)
         os.remove(db)

   def test_2_02(self):
      try:
         wordlist1 = wordlist_as_list(self.data['wordlist1'])
         tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         wl = tempfile.mktemp()
         db = tempfile.mktemp()
         tree.save(wl=wl, db=db)
         self.failUnless(os.path.exists(wl))
         self.failUnless(os.path.exists(db))
         self.failUnless(wordlist1 == wordlist_as_list(wl))
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(wordlist1 == tree1.list())
         self.failUnless(wordlist1 == tree2.list())
         self.failUnless(wordlist1 == tree3.list())
      finally:
         os.remove(wl)
         os.remove(db)

   def test_2_03(self):
      try:
         tree = TernarySearchTree(type=ON_DISK)
         wl = tempfile.mktemp()
         db = tempfile.mktemp()
         tree.save(wl=wl, db=db)
         self.failUnless(os.path.exists(wl))
         self.failUnless(os.path.exists(db))
         self.failUnless(os.stat(wl)[6] == 0)
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(tree1.list() == [])
         self.failUnless(tree2.list() == [])
         self.failUnless(tree3.list() == [])
      finally:
         os.remove(wl)
         os.remove(db)

   def test_2_04(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_2_05(self):
      try:
         db = tempfile.mktemp()
         shutil.copyfile(self.data['wordlist1.tdb'], db)
         wordlist1 = wordlist_as_list(self.data['wordlist1'])
         tree1 = TernarySearchTree(type=ON_DISK, db=db)
         try:
            tree1.save(db=db)
         except TreeError: pass
         else: self.fail("Expected TreeError")
      finally:
         os.remove(db)

   def test_2_06(self):
      try:
         dummy = tempfile.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         try:
            tree1.save(wl=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(wl=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(wl=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_07(self):
      try:
         dummy = tempfile.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         try:
            tree1.save(wl=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(wl=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(wl=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_08(self):
      try:
         dummy = tempfile.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
         try:
            tree1.save(wl=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(wl=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(wl=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_09(self):
      try:
         dummy = tempfile.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
         try:
            tree1.save(db=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(db=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(db=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_10(self):
      try:
         wordlist1 = wordlist_as_list(self.data['wordlist1'])
         tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         wl = tempfile.mktemp()
         db = tempfile.mktemp()
         tree.compress()
         tree.save(wl=wl, db=db)
         self.failUnless(wordlist1 == wordlist_as_list(wl))
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(wordlist1 == tree1.list())
         self.failUnless(wordlist1 == tree2.list())
         self.failUnless(wordlist1 == tree3.list())
      finally:
         os.remove(wl)
         os.remove(db)


##############
# Test3 class
##############

class Test3(unittest.TestCase):

   """Test 3: Searching for matches."""

   def setUp(self):
      try:
         self.data = find_data()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      pass

   def test_3_01(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))
      tree.compress()
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))

   def test_3_02(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))
      tree.compress
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))

   def test_3_03(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_3_04(self):
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))

   def test_3_05(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])
      tree.compress()
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])

   def test_3_06(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])
      tree.compress()
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])

   def test_3_07(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_3_08(self):
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])

   def test_3_09(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])
      tree.compress()
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])

   def test_3_10(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])
      tree.compress()
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])

   def test_3_11(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_3_12(self):
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])


##############
# Test4 class
##############

class Test4(unittest.TestCase):

   """Test 4: Modifying trees."""

   def setUp(self):
      try:
         self.data = find_data()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      pass

   def test_4_01(self):
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      try:
         tree.add(words_0_04a[1])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_4_02(self):
      # Removed along with support for temporary databases on disk
      pass

   def test_4_03(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_04(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_05(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)

   def test_4_06(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)

   def test_4_07(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_08(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_09(self):
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)

   def test_4_10(self):
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)


#######################################################################
# Helper functions
#######################################################################

def find_data():
   """Returns a dictionary of locations for wordlist1/wordlist1.tdb and wordlist2/wordlist2.tdb."""
   data = { }
   for dir in [ './data', './unittest/data' ]:
      found = True
      for file in [ 'wordlist1', 'wordlist1.tdb', 'wordlist2', 'wordlist2.tdb' ]:
         if not os.path.exists(os.path.join(dir, file)):
            found = False
            break
         else:
            data[file] = os.path.join(dir, file)
      if found:
         break
   if not found:
      raise Exception
   return data

def wordlist_as_list(wordlist):
   """Returns a list of words based on a wordlist on disk."""
   list = []
   for word in xreadlines.xreadlines(open(wordlist)):
      list.append(word[:-1])
   return list


#######################################################################
# Suite definition
#######################################################################

def suite():
   """Returns a suite containing all the test cases in this module."""
   return(unittest.makeSuite((Test1, 
                              Test2,
                              Test3,
                              Test4,
                              Test5)))


########################################################################
# Module entry point
########################################################################

# When this module is executed from the command-line, run its tests
if __name__ == '__main__':
   unittest.main()
