# -*- coding: iso-8859-1 -*-
# vim: set ft=python ts=3 sw=3 expandtab:
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
#              C E D A R
#          S O L U T I O N S       "Software done right."
#           S O F T W A R E
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Copyright (c) 2003-2005 Kenneth J. Pronovici.
# All rights reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# Version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Copies of the GNU General Public License are available from
# the Free Software Foundation website, http://www.gnu.org/.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
#
# Author   : Kenneth J. Pronovici <pronovic@ieee.org>
# Language : Python (>= 2.3)
# Project  : WordUtils
# Revision : $Id: treetests.py,v 1.13 2003/07/16 05:23:04 pronovic Exp $
# Purpose  : Unit tests for WordUtils/tree.py.
#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

########################################################################
# Module documentation
########################################################################

"""
Unit tests for WordUtils/tree.py.

Background
==========

   Python unittest conventions usually have tests named something like
   testListInsert() or similar.  This makes sense, but with the kind of tests
   I'm doing, I don't want to end up with function names like
   testLoadDictionaryAndWriteToDiskAndReadBackInAndValidate().  It's klunky.
   :-)

   Instead, I will provide this test plan, and then functions will be named
   according to their test plan item.  This is more like the way I've done my
   own regression testing in the past and I think for the most part, it will be
   just as easy to follow.

   This test plan is REALLY detailed, almost too detailed.  I feel that I have
   to be this detailed to adequately test the functionality.  Note that this
   test plan is very much similar to the plan for testing dawg.py, since the
   interface to the TernarySearchTree object almost identical to the interface
   to the Dawg object.

   Note that below, the extension .tdb stands for "ternary tree database".
   This is different that .ddb, which stands for "dawg database".  Also note
   that if the database format changes, you'll have to "bootstrap" the test by
   using the object to create the database files on disk, and then run the test
   with the newly-created database files.  I don't like this, but there's no
   good way around it.


Setup Notes
===========

   This unit test was originally in a different form, and some of the setup
   steps were listed explicitly and referred to in the test definitions.  You
   may find reference to the following steps in the tests below::

       0.01  Attempt to find the unittest data directory.  It should
             either be found at ./data or ./test/data.  If it is
             not found, issue a warning.

       0.02  Find the path to the wordlist1, wordlist1.tdb, wordlist2
             and wordlist2.tdb files in the data directory.  Each pair
             of files represents the same list of words.

       0.03  Load each wordlist into an array of words, for later use
             in comparison.

       0.04  Choose two sets of words.  The first set of words should
             not exist in wordlist1/database1.  The second set of
             words should exist in wordlist1/database1.  Save these
             sets of words for later use.

       0.05  Choose a set of patterns for which no words exist in
             wordlist1/database1.  Save this set of patterns for
             later use.

       0.06  Choose a set of patterns for which some words exist
             in wordlist1/database1.  Create a Python dictionary
             associating each pattern to the words in wordlist1/
             database1 which match it.  Save this dictionary for
             later use.

Debugging
=========

   Debugging in here is DAMN complicated.  If you have a test::

      def test():
        try:
           # stuff
        finally:
           # remove files

   you may mysteriously have the 'stuff' fail, and you won't get any exceptions
   reported to you.  The best thing to do if you get strange situations like
   this is to move 'stuff' out of the try block - that will usually clear
   things up.

   Run pychecker against this code using the -l option to ignore unused local
   variables - in many cases, local variables in here are not really "used",
   they're just created for testing purposes.

@author Kenneth J. Pronovici <pronovic@ieee.org>
"""


########################################################################
# Import modules and do runtime validations
########################################################################

# Standard modules
import sys
import os
import tempfile
import shutil
import unittest

# WordUtils modules
from WordUtils.tree import TernarySearchTree, TreeError, IN_MEMORY, ON_DISK


#######################################################################
# Module-wide configuration and constants
#######################################################################

# Words not in wordlist1
words_0_04a   = [ 'anecdotal', 'balfour', 'bolton', 'carcinogen', 'cochlea', 
                  'cowhide', 'desecrater', 'edmonds', 'extrusion', 'fringe', 
                  'growth', 'honky', 'iniquitous', 'knack', 'lumen', 'midget', 
                  'nightmarish', 'paregoric', 'podium', 'queer', 'rsvp', 'shaw', 
                  'spermatozoa', 'switchboard', 'tote', 'vest', 'xi' ]

# Words in wordlist1
words_0_04b   = [ 'alumnus', 'asteroid', 'bellatrix', 'brandenburg', 'caret',
                  'clasp', 'continuity', 'dabble', 'dilapidate', 'effusion',
                  'exhilarate', 'fluster', 'gestalt', 'handcuff', 'horrible',
                  'inexorable', 'journeymen', 'lengthwise', 'malpractice', 
                  'milton', 'nest', 'orville', 'perturbate', 'practitioner', 
                  'racy', 'ron', 'sedan', 'sleety', 'statuette', 'tabloid', 
                  'toni', 'urge', 'wet' ]
 
# Patterns not in wordlist1
patterns_0_05 = [ '.a.b.c', 'zz.al.', 'word...word', 'y..tta', 'bogus', 's.tuff' ] + words_0_04a

# Patterns in worldlist1
patterns_0_06 = { '.'        : [ 'a', 'k' ], 
                  '..'       : [ 'ad', 'el', 'ny', 'ph' ],
                  '...'      : [ 'bet', 'cdc', 'cia', 'cop', 'etc', 'fag', 'fee', 'her', 
                                 'imp', 'lim', 'mit', 'nay', 'ron', 'sat', 'spy', 'wet' ],
                  '.ac.'     : [ 'racy' ],
                  '.car.'    : [ 'scary' ],
                  '..r..'    : [ 'caret', 'curio', 'perez', 'surah', 'virgo' ],
                  'a...'     : [ 'able', 'aida', 'alex', 'aver', 'axis' ],
                  '..n'      : [ 'ron' ],
                  '.ee.'     : [ 'veer', 'week' ],
                  'b..e'     : [ 'byte' ],
                  't.......' : [ 'tanzania', 'tranquil', 'twilight' ],
                  '...a....' : [ 'alkaloid', 'almagest', 'escalate', 'fleawort', 'hilarity', 'strategy' ],
                  '.....c..' : [ 'pussycat', 'vertices' ],
                  '.....c'   : [ 'atypic', 'idetic', 'sextic' ],
                  'veer'     : [ 'veer' ] }


#######################################################################
# Utility functions
#######################################################################

def findData():
   """Returns a dictionary of locations for wordlist1/wordlist1.tdb and wordlist2/wordlist2.tdb."""
   data = { }
   for d in [ './data', './test/data' ]:
      found = True
      for f in [ 'wordlist1', 'wordlist1.tdb', 'wordlist2', 'wordlist2.tdb' ]:
         if not os.path.exists(os.path.join(d, f)):
            found = False
            break
         else:
            data[f] = os.path.join(d, f)
      if found:
         break
   if not found:
      raise Exception
   return data

def wordlistAsList(wordlist):
   """Returns a list of words based on a wordlist on disk."""
   l = []
   for word in open(wordlist):
      l.append(word[:-1])
   return l

def mktemp(dir=None):
   """
   Creates a temporary file and returns its name.

   This emulates the mktemp() function.  The returned file name has been
   removed from the filesystem, so it will not exist.  This is safe as long as
   you create the file in a temporary directory created with C{mkdtemp}.
   
   @param dir: Directory in which to create the file
   @return: Name of temporary file.
   """
   if dir is not None:
      f = tempfile.mkstemp(dir=dir)
   else:
      f = tempfile.mkstemp()
   os.close(f[0])
   if os.path.exists(f[1]):
      os.remove(f[1])
   return f[1]

def removedir(tree):
   """
   Recursively removes an entire directory.
   This is basically taken from an example on python.com.
   @param tree: Directory tree to remove.
   @raise ValueError: If a path cannot be encoded properly.
   """
   for root, dirs, files in os.walk(tree, topdown=False):
      for name in files:
         path = os.path.join(root, name)
         if os.path.islink(path):
            os.remove(path)
         elif os.path.isfile(path):
            os.remove(path)
      for name in dirs:
         path = os.path.join(root, name)
         if os.path.islink(path):
            os.remove(path)
         elif os.path.isdir(path):
            os.rmdir(path)
   os.rmdir(tree)


#######################################################################
# Test Case Classes
#######################################################################

##############
# Test1 class
##############

class Test1(unittest.TestCase):

   """Test 1: Loading and traversing trees; list representation."""

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.data = findData()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      removedir(self.tmpdir)

   def mktemp(self):
      return mktemp(dir=self.tmpdir)
         
   def test_1_01(self):
      """
      Test 1.01.
      Create an instance of the TernarySearchTree object.  Do not pass any
      arguments to the constructor.  Check that the object's type is set to
      IN_MEMORY.
      """
      tree = TernarySearchTree()
      self.failUnless(tree.type == IN_MEMORY)

   def test_1_02(self):
      """
      Test 1.02.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY but do not pass any other arguments to the
      constructor.  Check that the object's type is set to IN_MEMORY.
      """
      tree = TernarySearchTree(type=IN_MEMORY)
      self.failUnless(tree.type == IN_MEMORY)

   def test_1_03(self):
      """
      Test 1.03.
      Create an instance of the TernarySearchTree object.  Use type=ON_DISK
      but do not pass any other arguments to the constructor.  Check that
      the object's type is set to ON_DISK.
      """
      tree = TernarySearchTree(type=ON_DISK)
      self.failUnless(tree.type == ON_DISK)

   def test_1_04(self):
      """
      Create an instance of the TernarySearchTree object.  Use type=BAD but
      do not pass any other arguments to the constructor.  Ensure that a
      TreeError() exception is raised, because trees may only be
      constructed with a type of either IN_MEMORY or ON_DISK.
      """
      try:
         TernarySearchTree(type='bad')
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_05(self):
      """
      Test 1.05.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in both the path to the wordlist1 and the
      path to the database found in step 0.02.  Ensure that a TreeError()
      exception is raised, since objects may be initialized with either a
      wordlist or a database but not both.
      """
      try:
         TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'], db=self.data['wordlist1.tdb'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_06(self):
      """
      Test 1.06.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in both the path to the wordlist1 and the path
      to the database1 found in step 0.02.  Ensure that a TreeError()
      exception is raised, since objects may be initialized with either a
      wordlist or a database but not both.
      """
      try:
         TernarySearchTree(type=ON_DISK, wl=self.data['wordlist1'], db=self.data['wordlist1.tdb'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_07(self):
      """
      Test 1.07.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Ensure that the object's wordlist is set to the passed in
      path, and that the object's database is None.  Use the list() method
      to traverse the tree, building an array of words.  Compare the array
      with the one built in step 0.03.  The two arrays should match.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      self.failUnless(tree.wl == self.data['wordlist1'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist1)

   def test_1_08(self):
      """
      Test 1.08.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  Ensure that the object's database is set to the passed in
      path, and that the object's wordlist is None.  Use the list() method
      to traverse the tree, building an array of words.  Compare the array
      with the one built in step 0.03.  The two arrays should match.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_09(self):
      """
      Test 1.09.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in the path to the database1 found in step
      0.02.  Ensure that the object's database is set to the passed in
      path, and that the object's wordlist is None.  Use the list() method
      to traverse the tree, building an array of words.  Compare the array
      with the one built in step 0.03.  The two arrays should match.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_10(self):
      """
      Test 1.10.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_1_11(self):
      """
      Test 1.11.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method, and pass in both the path to the wordlist1 and the
      path to the database1 found in step 0.02.  Ensure that a TreeError()
      exception is raised, since loadFile() may be called with either a
      wordlist or a database but not both.
      """
      tree = TernarySearchTree(type=IN_MEMORY)
      try:
         tree.loadFile(wl=self.data['wordlist1'], db=self.data['wordlist1.tdb'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_12(self):
      """
      Test 1.12.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method, and pass in the path to the wordlist1 found in
      step 0.02.  Ensure that the object's wordlist is set to the passed in
      path, and that the object's database is None.  Use the list() method
      to traverse the tree, building an array of words.  Compare the array
      with the one built in step 0.03.  The two arrays should match.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      self.failUnless(tree.wl == self.data['wordlist1'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist1)

   def test_1_13(self):
      """
      Test 1.13.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method, and pass in the path to the database1 found in
      step 0.02.  Ensure that the object's database is set to the passed in
      path, and that the object's wordlist is None.  Use the list() method
      to traverse the tree, building an array of words.  Compare the array
      with the one built in step 0.03.  The two arrays should match.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_14(self):
      """
      Test 1.14.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the database1 found in step
      0.02.  Ensure that the object's database is set to the passed in
      path, and that the object's wordlist is None.  Use the list() method
      to traverse the tree, building an array of words.  Compare the array
      with the one built in step 0.03.  The two arrays should match.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist1.tdb'])
      self.failUnless(tree.list() == wordlist1)

   def test_1_15(self):
      """
      Test 1.15.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_1_16(self):
      """
      Test 1.16.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the wordlist1 found in step
      0.02.  Save off the values of the object's root, wordlist and
      database members.  Call the loadFile() method again, and this time,
      pass in the path to a wordlist that does not exist.  Ensure that an
      exception is raised.  Ensure that after the loadFile() call has
      completed, the object's root, wordlist, and database members have not
      been changed.  Repeat for a database that does not exist.
      """
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      root = tree.root
      wl = tree.wl
      db = tree.db
      try:
         tree.loadFile(wl=self.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)
      try:
         tree.loadFile(db=self.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)

   def test_1_17(self):
      """
      Test 1.17.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the database1 found in step
      0.02.  Save off the values of the object's root, wordlist and
      database members.  Call the loadFile() method again, and this time,
      pass in the path to a wordlist that does not exist.  Ensure that an
      exception is raised.  Ensure that after the loadFile() call has
      completed, the object's root, wordlist and database members have not
      been changed.  Repeat for a database that does not exist.
      """
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      root = tree.root
      wl = tree.wl
      db = tree.db
      try:
         tree.loadFile(wl=self.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)
      try:
         tree.loadFile(db=self.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)

   def test_1_18(self):
      """
      Test 1.18.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_1_19(self):
      """
      Test 1.19.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the database1 found in step
      0.02.  Save off the values of the object's root, wordlist and
      database members.  Call the loadFile() method again, and this time,
      pass in the path to a wordlist that does not exist.  Ensure that an
      exception is raised.  Ensure that after the loadFile() call has
      completed, the object's root, wordlist and database members have not
      been changed.  Repeat for a database that does not exist.
      """
      tree = TernarySearchTree(type=ON_DISK)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      root = tree.root
      wl = tree.wl
      db = tree.db
      try:
         tree.loadFile(wl=self.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)
      try:
         tree.loadFile(db=self.mktemp())
      except TreeError: pass
      else: self.fail("Expected TreeError")
      self.failUnless(root == tree.root)
      self.failUnless(wl == tree.wl)
      self.failUnless(db == tree.db)

   def test_1_20(self):
      """
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the wordlist1 found in step
      0.02.  Call the loadFile() method again, and this time, pass in the
      path to the wordlist2 found in step 0.02.  Ensure that the object's
      wordlist is set to the passed in path, and that the object's database
      is None.  Use the list() method to traverse the tree, building an
      array of words.  Compare the array with the one built in step 0.03.
      The two arrays should match.
      """
      wordlist2 = wordlistAsList(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      tree.loadFile(wl=self.data['wordlist2'])
      self.failUnless(tree.wl == self.data['wordlist2'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist2)

   def test_1_21(self):
      """
      Test 1.21.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the wordlist1 found in step
      0.02.  Call the loadFile() method again, and this time, pass in the
      path to the database2 found in step 0.02.  Ensure that the object's
      database is set to the passed in path, and that the object's wordlist
      is None.  Use the list() method to traverse the tree, building an
      array of words.  Compare the array with the one built in step 0.03.
      The two arrays should match.
      """
      wordlist2 = wordlistAsList(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(wl=self.data['wordlist1'])
      tree.loadFile(db=self.data['wordlist2.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db  == self.data['wordlist2.tdb'])
      self.failUnless(tree.list() == wordlist2)

   def test_1_22(self):
      """
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the database1 found in step
      0.02.  Call the loadFile() method again, and this time, pass in the
      path to the database2 found in step 0.02.  Ensure that the object's
      database is set to the passed in path, and that the object's wordlist
      is None.  Use the list() method to traverse the tree, building an
      array of words.  Compare the array with the one built in step 0.03.
      The two arrays should match.
      """
      wordlist2 = wordlistAsList(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      tree.loadFile(db=self.data['wordlist2.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db  == self.data['wordlist2.tdb'])
      self.failUnless(tree.list() == wordlist2)

   def test_1_23(self):
      """
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the database1 found in step
      0.02.  Call the loadFile() method again, and this time, pass in the
      path to the wordlist2 found in step 0.02.  Ensure that the object's
      wordlist is set to the passed in path, and that the object's database
      is None.  Use the list() method to traverse the tree, building an
      array of words.  Compare the array with the one built in step 0.03.
      The two arrays should match.
      """
      wordlist2 = wordlistAsList(self.data['wordlist2'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      tree.loadFile(wl=self.data['wordlist2'])
      self.failUnless(tree.wl == self.data['wordlist2'])
      self.failUnless(tree.db is None)
      self.failUnless(tree.list() == wordlist2)

   def test_1_24(self):
      """
      Test 1.24.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_1_25(self):
      """
      Test 1.25.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_1_26(self):
      """
      Test 1.26.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, but do not provide a wordlist or database.  Call the
      loadFile() method and pass in the path to the database1 found in step
      0.02.  Call the loadFile() method again, and this time, pass in the
      path to the database2 found in step 0.02.  Ensure that the object's
      database is set to the passed in path, and that the object's wordlist
      is None.  Use the list() method to traverse the tree, building an
      array of words.  Compare the array with the one built in step 0.03.
      The two arrays should match.
      """
      wordlist2 = wordlistAsList(self.data['wordlist2'])
      tree = TernarySearchTree(type=ON_DISK)
      tree.loadFile(db=self.data['wordlist1.tdb'])
      tree.loadFile(db=self.data['wordlist2.tdb'])
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db == self.data['wordlist2.tdb'])
      self.failUnless(tree.list() == wordlist2)

   def test_1_27(self):
      """
      Test 1.27.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_1_28(self):
      """
      Test 1.28.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and provide as the *database* the path to the
      wordlist1 found in step 0.02.  Ensure that an exception is raised,
      since a wordlist should not be a valid database.
      """
      try:
         TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1'])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_29(self):
      """
      Test 1.29.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in the path to the database1 found in step
      0.02.  Call the compress() method on the tree and ensure that a
      TreeError() exception is raised, since the compress() method can only
      be called on IN_MEMORY trees.
      """
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      try:
         tree.compress()
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_1_30(self):
      """
      Test 1.30.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Use the list() method to traverse the tree, building an array
      of words, and save off the result.  Call the compress() method on the
      tree, and then use the list() method again to build a second array of
      words.  Compare the two arrays, which should be identical.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      self.failUnless(tree.list() == wordlist1)
      tree.compress()
      self.failUnless(tree.list() == wordlist1)

   def test_1_31(self):
      """
      Test 1.31.
      Create an instance of the Dawg object.  Use type=ON_DISK.  Do not
      pass in any wordlist or database.  Load a list of words, and call the
      loadList() method.  Ensure that a DawgError() exception is raised,
      since ON_DISK dawgs may only be loaded from disk.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=ON_DISK)
      try:
         tree.loadList(wordlist1)
      except TreeError: pass
      else: self.fail("Expected TreeError");

   def test_1_32(self):
      """
      Test 1.32.
      Create an instance of the Dawg object.  Use type=IN_MEMORY.  Do not
      pass in any wordlist or database.  Load a list of words, and call the
      loadList() method.   Ensure that the object's database is None and
      its wordlist is None.  Use the list() method to traverse the dawg,
      building an array of words, and save off the result.  Check that this
      result is the same as the original passed-in wordlist.  Call the
      compress() method on the dawg, and then use the list() method again
      to build a second array of words.  Compare the two arrays, which
      should be identical. Then, call the compress method a second time.
      The results should be the same, since it should be safe to compress
      an already- compressed dawg.
      """
      wordlist1 = wordlistAsList(self.data['wordlist1'])
      tree = TernarySearchTree(type=IN_MEMORY)
      tree.loadList(wordlist1)
      self.failUnless(tree.type == IN_MEMORY)
      self.failUnless(tree.wl is None)
      self.failUnless(tree.db is None)
      list1 = tree.list()
      self.failUnless(list1 == wordlist1)
      tree.compress()
      list2 = tree.list()
      self.failUnless(list1 == list2)
      list2 = tree.list()
      self.failUnless(list1 == list2)


##############
# Test2 class
##############

class Test2(unittest.TestCase):

   """Test 2: Saving tree objects to disk."""

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.data = findData()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      removedir(self.tmpdir)

   def mktemp(self):
      return mktemp(dir=self.tmpdir)

   def test_2_01(self):
      """
      Test 2.01.
      Create an instance of the TernarySeachTree object.  Use
      type=IN_MEMORY, but do not provide a wordlist or database.  Call the
      save() method and pass in a path for the wordlist and database.  Make
      sure that the wordlist and database do not already exist on disk.
      Ensure that the wordlist and database are created.  The wordlist
      should be an empty file.  Create four new instances of the
      TernarySeachTree object.  For the first one, use type=IN_MEMORY, and
      provide the newly-saved wordlist, for the second one, use
      type=ON_DISK and provide the newly-saved database, for the third one,
      use type=ON_DISK and provide the newly-saved wordlist, and for the
      fourth one, use type=IN_MEMORY and provide the newly saved database.
      Ensure that the list() method returns an empty list for all new
      objects.  Remove the newly-created wordlist and database.
      """
      try:
         tree = TernarySearchTree(type=IN_MEMORY)
         wl = self.mktemp()
         db = self.mktemp()
         tree.save(wl=wl, db=db)
         self.failUnless(os.path.exists(wl))
         self.failUnless(os.path.exists(db))
         self.failUnless(os.stat(wl)[6] == 0)
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(tree1.list() == [])
         self.failUnless(tree2.list() == [])
         self.failUnless(tree3.list() == [])
      finally:
         os.remove(wl)
         os.remove(db)

   def test_2_02(self):
      """
      Test 2.02.
      Create an instance of the TernarySeachTree object.  Use
      type=IN_MEMORY, and provide the wordlist1 found in step 0.02.  Call
      the save() method and pass in a path for the wordlist and database.
      Make sure that the wordlist and database do not already exist on
      disk.  Ensure that the wordlist and database are created.  Compare
      the wordlist1 from step 0.02 and the newly- created wordlist on disk.
      Ensure that they are identical.  Create four new instances of the
      TernarySeachTree object.  For the first one, use type=IN_MEMORY, and
      provide the newly-saved wordlist, for the second one, use
      type=ON_DISK and provide the newly-saved database, for the third one,
      use type= ON_DISK and provide the newly-saved wordlist, and for the
      fourth one, use type=IN_MEMORY and provide the newly saved database.
      Ensure that the list() method returns a list with the same contents
      for each of the three objects.  Remove the newly-created wordlist and
      database.
      """
      try:
         wordlist1 = wordlistAsList(self.data['wordlist1'])
         tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         wl = self.mktemp()
         db = self.mktemp()
         tree.save(wl=wl, db=db)
         self.failUnless(os.path.exists(wl))
         self.failUnless(os.path.exists(db))
         self.failUnless(wordlist1 == wordlistAsList(wl))
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(wordlist1 == tree1.list())
         self.failUnless(wordlist1 == tree2.list())
         self.failUnless(wordlist1 == tree3.list())
      finally:
         os.remove(wl)
         os.remove(db)

   def test_2_03(self):
      """
      Test 2.03.
      Create an instance of the TernarySeachTree object.  Use type=ON_DISK,
      but do not provide a wordlist or database.  Call the save() method
      and pass in a path for a wordlist and database.  Make sure that the
      wordlist and database do not already exist on disk.  Ensure that the
      wordlist and database are created.  The wordlist should be an empty
      file.  Create four new instances of the TernarySeachTree object.  For
      the first one, use type=IN_MEMORY, and provide the newly-saved
      wordlist, for the second one, use type=ON_DISK and provide the
      newly-saved database, for the third one, use type=ON_DISK and provide
      the newly-saved wordlist, and for the fourth one, use type=IN_MEMORY
      and provide the newly saved database.  Ensure that the list() method
      returns an empty list for all new objects.  Remove the newly-created
      wordlist and database.
      """
      try:
         tree = TernarySearchTree(type=ON_DISK)
         wl = self.mktemp()
         db = self.mktemp()
         tree.save(wl=wl, db=db)
         self.failUnless(os.path.exists(wl))
         self.failUnless(os.path.exists(db))
         self.failUnless(os.stat(wl)[6] == 0)
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(tree1.list() == [])
         self.failUnless(tree2.list() == [])
         self.failUnless(tree3.list() == [])
      finally:
         os.remove(wl)
         os.remove(db)

   def test_2_04(self):
      """
      Test 2.04.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_2_05(self):
      """
      Test 2.05.
      Make a copy of the database1 file found in step 0.02.  Create an
      instance of the TernarySeachTree object.  Use type=ON_DISK, and
      provide the copied database1 file.  Call the save() method and pass
      in a the path to the copied database1 file .  Ensure that an
      exception is raised, because the object is not allowed to overwrite
      its own datbase with a save.  (You are making a copy so this test
      can't trash the unittest suite.)  Remove the copied database file.
      """
      try:
         db = self.mktemp()
         shutil.copyfile(self.data['wordlist1.tdb'], db)
         tree1 = TernarySearchTree(type=ON_DISK, db=db)
         try:
            tree1.save(db=db)
         except TreeError: pass
         else: self.fail("Expected TreeError")
      finally:
         os.remove(db)

   def test_2_06(self):
      """
      Test 2.06.
      Create a dummy file on disk, with known contents.  Create an instance
      of the TernarySeachTree object.  Use type=IN_MEMORY, and provide the
      wordlist1 found in step 0.02.  Call the save() method and pass in a
      the path to the dummy file as the wordlist.  Ensure that an exception
      is raised, because the object is not allowed to overwrite a wordlist
      on disk if overwrite is False.  Call the save() method again, this
      time passing overwrite=False explicitly.  Ensure that the results are
      the same.  Call the save() method again, this time passing
      overwrite=True explicitly.  Ensure that the dummy file has been
      overwritten.  Remove the dummy file.
      """
      try:
         dummy = self.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         try:
            tree1.save(wl=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(wl=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(wl=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_07(self):
      """
      Test 2.07.
      Create a dummy file on disk, with known contents.  Create an instance
      of the TernarySeachTree object.  Use type=IN_MEMORY, and provide the
      wordlist1 found in step 0.02.  Call the save() method and pass in a
      the path to the dummy file as the database.  Ensure that an exception
      is raised, because the object is not allowed to overwrite a database
      on disk if overwrite is False.  Call the save() method again, this
      time passing overwrite=False explicitly.  Ensure that the results are
      the same.  Call the save() method again, this time passing
      overwrite=True explicitly.  Ensure that the dummy file has been
      overwritten.  Remove the dummy file.
      """
      try:
         dummy = self.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         try:
            tree1.save(wl=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(wl=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(wl=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_08(self):
      """
      Test 2.08.
      Create a dummy file on disk, with known contents.  Create an instance
      of the TernarySeachTree object.  Use type=ON_DISK, and provide the
      database1 found in step 0.02.  Call the save() method and pass in a
      the path to the dummy file as the wordlist.  Ensure that an exception
      is raised, because the object is not allowed to overwrite a wordlist
      on disk if overwrite is False.  Call the save() method again, this
      time passing overwrite=False explicitly.  Ensure that the results are
      the same.  Call the save() method again, this time passing
      overwrite=True explicitly.  Ensure that the dummy file has been
      overwritten.  Remove the dummy file.
      """
      try:
         dummy = self.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
         try:
            tree1.save(wl=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(wl=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(wl=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_09(self):
      """
      Test 2.09.
      Create a dummy file on disk, with known contents.  Create an instance
      of the TernarySeachTree object.  Use type=ON_DISK, and provide the
      database1 found in step 0.02.  Call the save() method and pass in a
      the path to the dummy file as the database.  Ensure that an exception
      is raised, because the object is not allowed to overwrite a database
      on disk if overwrite is False.  Call the save() method again, this
      time passing overwrite=False explicitly.  Ensure that the results are
      the same.  Call the save() method again, this time passing
      overwrite=True explicitly.  Ensure that the dummy file has been
      overwritten.  Remove the dummy file.
      """
      try:
         dummy = self.mktemp()
         open(dummy, "w").write("HELLO\n")
         self.failUnless("HELLO\n" == open(dummy, "r").readline())
         tree1 = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
         try:
            tree1.save(db=dummy)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         try:
            tree1.save(db=dummy, overwrite=False)
         except TreeError: pass
         else: self.fail("Expected TreeError")
         tree1.save(db=dummy, overwrite=True)
         self.failUnless("HELLO\n" != open(dummy, "r").readline())
      finally:
         os.remove(dummy)

   def test_2_10(self):
      """
      Test 2.10.
      Create an instance of the TernarySeachTree object.  Use
      type=IN_MEMORY, and provide the wordlist1 found in step 0.02.  Call
      the compress() method to compress the tree, and then call the save()
      method and pass in a path for the wordlist and database.  Make sure
      that the wordlist and database do not already exist on disk.  Ensure
      that the wordlist and database are created.  Compare the wordlist1
      from step 0.02 and the newly- created wordlist on disk.  Ensure that
      they are identical.  Create four new instances of the
      TernarySeachTree object.  For the first one, use type=IN_MEMORY, and
      provide the newly-saved wordlist, for the second one, use
      type=ON_DISK and provide the newly-saved database, for the third one,
      use type= ON_DISK and provide the newly-saved wordlist, and for the
      fourth one, use type=IN_MEMORY and provide the newly saved database.
      Ensure that the list() method returns a list with the same contents
      for each of the three objects.  Remove the newly-created wordlist and
      database.
      """
      try:
         wordlist1 = wordlistAsList(self.data['wordlist1'])
         tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
         wl = self.mktemp()
         db = self.mktemp()
         tree.compress()
         tree.save(wl=wl, db=db)
         self.failUnless(wordlist1 == wordlistAsList(wl))
         tree1 = TernarySearchTree(type=IN_MEMORY, wl=wl)
         tree2 = TernarySearchTree(type=ON_DISK, db=db)
         tree3 = TernarySearchTree(type=IN_MEMORY, db=db)
         self.failUnless(wordlist1 == tree1.list())
         self.failUnless(wordlist1 == tree2.list())
         self.failUnless(wordlist1 == tree3.list())
      finally:
         os.remove(wl)
         os.remove(db)


##############
# Test3 class
##############

class Test3(unittest.TestCase):

   """Test 3: Searching for matches."""

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.data = findData()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      removedir(self.tmpdir)

   def mktemp(self):
      return mktemp(dir=self.tmpdir)

   def test_3_01(self):
      """
      Test 3.01.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Using the two sets of words developed in step 0.04, ensure
      that the search() method returns appropriate results.  The words
      which are in the wordlist1 should be found, and the words which are
      not in the wordlist1 should not be found.  Repeat all tests after
      calling the compress() method.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))
      tree.compress()
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))

   def test_3_02(self):
      """
      Test 3.02.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  Using the two sets of words developed in step 0.04, ensure
      that the search() method returns appropriate results.  The words
      which are in the database1 should be found, and the words which are
      not in the database1 should not be found.  Repeat all tests after
      calling the compress() method.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))
      tree.compress()
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))

   def test_3_03(self):
      """
      Test 3.03.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_3_04(self):
      """
      Test 3.04.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in the path to the database1 found in step
      0.02.  Using the two sets of words developed in step 0.04, ensure
      that the search() method returns appropriate results.  The words
      which are in the database1 should be found, and the words which are
      not in the database1 should not be found.
      """
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      for word in words_0_04a:
         self.failUnless(not tree.search(word))
      for word in words_0_04b:
         self.failUnless(tree.search(word))

   def test_3_05(self):
      """
      Test 3.05.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Using the set of patterns developed in step 0.05, ensure that
      the patternSearch() method returns appropriate results.  The method
      should return an empty list for each of the patterns.  Repeat all
      tests after calling the compress() method.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])
      tree.compress()
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])

   def test_3_06(self):
      """
      Test 3.06.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  Using the set of patterns developed in step 0.05, ensure that
      the patternSearch() method returns appropriate results.  The method
      should return an empty list for each of the patterns.  Repeat all
      tests after calling the compress() method.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])
      tree.compress()
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])

   def test_3_07(self):
      """
      Test 3.07.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_3_08(self):
      """
      Test 3.08.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in the path to the database1 found in step
      0.02.  Using the set of patterns developed in step 0.05, ensure that
      the patternSearch() method returns appropriate results.  The method
      should return an empty list for each of the patterns.
      """
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_05:
         self.failUnless(tree.patternSearch(pattern) == [])

   def test_3_09(self):
      """
      Test 3.09.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Using the Python dictionary mapping pattern to list of
      matching words developed in step 0.06, ensure that the
      patternSearch() method returns appropriate results.  For each
      pattern, the method should return a list that matches the
      corresponding list in the Python dictionary.  Repeat all tests after
      calling the compress() method.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])
      tree.compress()
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])

   def test_3_10(self):
      """
      Test 3.10.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  Using the Python dictionary mapping pattern to list of
      matching words developed in step 0.06, ensure that the
      patternSearch() method returns appropriate results.  For each
      pattern, the method should return a list that matches the
      corresponding list in the Python dictionary.  Repeat all tests after
      calling the compress() method.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])
      tree.compress()
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])

   def test_3_11(self):
      """
      Test 3.11.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_3_12(self):
      """
      Test 3.12.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in the path to the database1 found in step
      0.02.  Using the Python dictionary mapping pattern to list of
      matching words developed in step 0.06, ensure that the
      patternSearch() method returns appropriate results.  For each
      pattern, the method should return a list that matches the
      corresponding list in the Python dictionary.
      """
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      for pattern in patterns_0_06.keys():
         self.failUnless(tree.patternSearch(pattern) == patterns_0_06[pattern])


##############
# Test4 class
##############

class Test4(unittest.TestCase):

   """Test 4: Modifying trees."""

   def setUp(self):
      try:
         self.tmpdir = tempfile.mkdtemp()
         self.data = findData()
      except:
         self.fail("Unable to find some required unittest data.")

   def tearDown(self):
      removedir(self.tmpdir)

   def mktemp(self):
      return mktemp(dir=self.tmpdir)

   def test_4_01(self):
      """
      Test 4.01.
      Create an instance of the TernarySearchTree object.  Use
      type=ON_DISK, and pass in the path to the database1 found in step
      0.02.  From the list developed in step 0.04, choose a word that does
      not exist in database1.  Call the add() method using the word.
      Ensure that an exception is raised, since on-disk trees are
      read-only.
      """
      tree = TernarySearchTree(type=ON_DISK, db=self.data['wordlist1.tdb'])
      try:
         tree.add(words_0_04a[1])
      except TreeError: pass
      else: self.fail("Expected TreeError")

   def test_4_02(self):
      """
      Test 4.02.
      Removed along with support for temporary databases on disk.
      """
      pass

   def test_4_03(self):
      """
      Test 4.03.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  From the list developed in step 0.04, choose a word that does
      not exist in wordlist1.  Save off the contents of the tree as a list.
      Call the add() method using the word.  Get the contents of the new
      tree as a list.  Ensure that the new word has been added to the list,
      and that otherwise the saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_04(self):
      """
      Test 4.04.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  From the list developed in step 0.04, choose a word that does
      not exist in database1.  Save off the contents of the tree as a list.
      Call the add() method using the word.  Get the contents of the new
      tree as a list.  Ensure that the new word has been added to the list,
      and that otherwise the saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_05(self):
      """
      Test 4.05.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  From the list developed in step 0.04, choose a word that
      exists in wordlist1.  Save off the contents of the tree as a list.
      Call the remove() method using the word.  Get the contents of the new
      tree as a list.  Ensure that the word has been removed from the list,
      and that otherwise the saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)

   def test_4_06(self):
      """
      Test 4.06.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  From the list developed in step 0.04, choose a word that
      exists in database1.  Save off the contents of the tree as a list.
      Call the remove() method using the word.  Get the contents of the new
      tree as a list.  Ensure that the new word has been removed from the
      list, and that otherwise the saved-off list and the new list are
      identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)

   def test_4_07(self):
      """
      Test 4.07.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Call the compress() method for the tree.  From the list
      developed in step 0.04, choose a word that does not exist in
      wordlist1.  Save off the contents of the tree as a list.  Call the
      add() method using the word.  Get the contents of the new tree as a
      list.  Ensure that the new word has been added to the list, and that
      otherwise the saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_08(self):
      """
      Test 4.08.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  Call the compress() method for the tree.  From the list
      developed in step 0.04, choose a word that does not exist in
      database1.  Save off the contents of the tree as a list.  Call the
      add() method using the word.  Get the contents of the new tree as a
      list.  Ensure that the new word has been added to the list, and that
      otherwise the saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(not words_0_04a[1] in list_pre)
      tree.add(words_0_04a[1])
      list_post = tree.list()
      self.failUnless(words_0_04a[1] in list_post)
      list_post.remove(words_0_04a[1])
      self.failUnless(list_pre == list_post)

   def test_4_09(self):
      """
      Test 4.09.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the wordlist1 found in step
      0.02.  Call the compress() method for the tree.  From the list
      developed in step 0.04, choose a word that exists in wordlist1.  Save
      off the contents of the tree as a list.  Call the remove() method
      using the word.  Get the contents of the new tree as a list.  Ensure
      that the word has been removed from the list, and that otherwise the
      saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, wl=self.data['wordlist1'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)

   def test_4_10(self):
      """
      Test 4.10.
      Create an instance of the TernarySearchTree object.  Use
      type=IN_MEMORY, and pass in the path to the database1 found in step
      0.02.  Call the compress() method for the tree.  From the list
      developed in step 0.04, choose a word that exists in database1.  Save
      off the contents of the tree as a list.  Call the remove() method
      using the word.  Get the contents of the new tree as a list.  Ensure
      that the new word has been removed from the list, and that otherwise
      the saved-off list and the new list are identical.
      """
      tree = TernarySearchTree(type=IN_MEMORY, db=self.data['wordlist1.tdb'])
      tree.compress()
      list_pre = tree.list()
      self.failUnless(words_0_04b[1] in list_pre)
      tree.remove(words_0_04b[1])
      list_post = tree.list()
      self.failUnless(not words_0_04b[1] in list_post)
      list_pre.remove(words_0_04b[1])
      self.failUnless(list_pre == list_post)


#######################################################################
# Suite definition
#######################################################################

def suite():
   """Returns a suite containing all the test cases in this module."""
   return unittest.TestSuite((
                              unittest.makeSuite(Test1, 'test'),
                              unittest.makeSuite(Test2, 'test'),
                              unittest.makeSuite(Test3, 'test'),
                              unittest.makeSuite(Test4, 'test'),
                            ))


########################################################################
# Module entry point
########################################################################

# When this module is executed from the command-line, run its tests
if __name__ == '__main__':
   unittest.main()

